{******************************************************************************}
{ FileName............: MajorDvbRecording                                      }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 0.03                                                   }
{------------------------------------------------------------------------------}
{  Recordings                                                                  }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  0.01   20041211 - Initial release                                           }
{  0.02   20050111 - Bugfix GetSetting returning not all setting               }
{  0.03   20050611 - Bugfix concerning valid PID values                        }
{                                                                              }
{ To know:                                                                     }
{ Format is 'RR DDDDDDDD SS:SS:SS EE:EE:EE [PPPP(name), PPPP(name), ....       }
{                    ...., PPPP(name)] LLL FFFF TUNE NAME                      }
{              RR        = Recording identifier (HEX, FF if not yet recording) }
{              DD...     = Date in YYYYMMDD format                             }
{              SS        = Start time                                          }
{              EE        = Stop time                                           }
{              PPPP      = PID                                                 }
{               (name)   = Optional name of PID                                }
{              LLL       = List index                                          }
{              FFFF      = Favourite index                                     }
{              TUNE      = Tune information                                    }
{                            SAT_FREQ_POL_SR format, eg. 1_11837_H_27500       }
{              NAME     = Name of program                                      }
{******************************************************************************}
unit MajorDvbRecording;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Grids, ComCtrls;

type
  TPidList = record
    Pid: Word;
    PidName: string[10];
  end;
  TPidLists = array[0..31] of TPidList;

  PRecordItem = ^TRecordItem;
  TRecordItem = record
    Active: Boolean; // True if active, False if not in use
    RecordId: Byte; // Identifier of recording ($FF if none)
    Miscellaneous: Integer;
    StartTime: TDateTime; // Start date/time
    StopTime: TDateTime; // Stop date/time
    Pids: TPidLists; // Pids to record
    ListIndex: Integer; // List to use
    FavouriteIndex: Integer; // Item in list to use
    Tuning: SHortString; // Tuning information
    ProgramName: ShortString; // Program name (informative only)
  end;

  TRecordItems = array[0..31] of TRecordItem;

  TfrmRecording = class(TForm)
    btnOK: TButton;
    btnDiscard: TButton;
    Shape1: TShape;
    sgRecordings: TStringGrid;
    btnUp: TButton;
    btnDown: TButton;
    btnRemove: TButton;
    btnRemoveAll: TButton;
    grpRecording: TGroupBox;
    Shape3: TShape;
    Shape4: TShape;
    dtStartRecording: TDateTimePicker;
    dtStopRecording: TDateTimePicker;
    Image1: TImage;
    Image2: TImage;
    edtPids: TEdit;
    Label1: TLabel;
    Label2: TLabel;
    btnRevert: TButton;
    dtStartRecordingDate: TDateTimePicker;
    procedure SetSetting(Index: Integer; Value: ShortString);
    function GetSetting(Index: Integer): ShortString;
    procedure btnOKClick(Sender: TObject);
    procedure btnDiscardClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRemoveClick(Sender: TObject);
    procedure btnUpClick(Sender: TObject);
    procedure btnDownClick(Sender: TObject);
    function CheckPids(CheckData: ShortString): ShortString;
    procedure btnRevertClick(Sender: TObject);
    procedure sgRecordingsClick(Sender: TObject);
    procedure edtPidsKeyPress(Sender: TObject; var Key: Char);
    procedure btnRemoveAllClick(Sender: TObject);
    procedure dtStartRecordingDateChange(Sender: TObject);
    procedure dtStartRecordingChange(Sender: TObject);
    procedure dtStopRecordingChange(Sender: TObject);
    procedure edtPidsExit(Sender: TObject);
  private
    { Private declarations }
    FOrgSettings: array[0..99] of TRecordItem;
    FSettings: array[0..99] of TRecordItem;
    FOrgRowCount: Integer;
    FExitCallback: TNotifyEvent;
    FAccepted: Boolean;
    procedure Revert;
    function GetSettingChanged: Boolean;
  public
    { Public declarations }
    property Setting[Index: Integer]: ShortString read GetSetting write
    SetSetting;
    property ExitNotify: TNotifyEvent read FExitCallback write FExitCallback;
    property SettingChanged: Boolean read GetSettingChanged;
    property Accepted: Boolean read FAccepted;
  end;

function ExtractFromRecordingDefinition(ExtractString: ShortString):
  TRecordItem;

function ConstructRecordingDefinition(Definition: TRecordItem): ShortString;

function ExtractFromPidDefinition(ExtractString: ShortString): TPidLists;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
  Params  : <ExtractString>   String to extract information from
  Returns : <Result>          Extracted items from string
                              .Active set to False if error, else True

  Descript: Extract information from string
  Notes   : RR DDDDDDDD SS:SS:SS EE:EE:EE [PPPP(name), ..] LLL FFFF TUNE NAME
            Pids are placed in ascending order and double entries are removed
 ------------------------------------------------------------------------------}

function ExtractFromRecordingDefinition(ExtractString: ShortString):
  TRecordItem;
var
  CheckString: ShortString;
  Index: Integer;
  Error: Integer;
  ThePos: Integer;
  StartDate: TDateTime;
begin
  // Initial default results
  Result.Active := False;

  Result.RecordId := $FF;
  Result.StartTime := Date + EncodeTime(0, 0, 0, 0);
  Result.StopTime := EncodeTime(0, 0, 0, 0);
  for Index := Low(Result.Pids) to High(Result.Pids) do
  begin
    Result.Pids[Index].Pid := $FFFF;
    Result.Pids[Index].PidName := '';
  end;
  Result.ListIndex := -1;
  Result.FavouriteIndex := -1;
  Result.Tuning := '';
  Result.ProgramName := '';

  // Record identifier
  ExtractString := Trim(ExtractString);
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  Val('$' + CheckString, Result.RecordId, Error);
  if Error <> 0 then
    Exit;
  // Date
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  try
    StartDate := EncodeDate(StrToInt(Copy(CheckString, 1, 4)),
      StrToInt(Copy(CheckString, 5, 2)), StrToInt(Copy(CheckString, 7, 2)));
  except
    Exit;
  end;
  // Start time
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  try
    Result.StartTime := StartDate + EncodeTime(StrToInt(Copy(CheckString, 1,
      2)),
      StrToInt(Copy(CheckString, 4, 2)), StrToInt(Copy(CheckString, 7, 2)), 0);
  except
    Exit;
  end;
  // Stop time
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  try
    Result.StopTime := StartDate + EncodeTime(StrToInt(Copy(CheckString, 1, 2)),
      StrToInt(Copy(CheckString, 4, 2)), StrToInt(Copy(CheckString, 7, 2)), 0);
  except
    Exit;
  end;
  // Correct for a single day
  if Result.StopTime < Result.StartTime then
    Result.StopTime := Result.StopTime + 1;
  // PIDs
  ThePos := Pos('[', ExtractString);
  if ThePos <> 1 then
    Exit;
  ThePos := Pos(']', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 2, ThePos - 2));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  Result.Pids := ExtractFromPidDefinition(CheckString);
  // List index
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  Val(CheckString, Result.ListIndex, Error);
  if Error <> 0 then
    Exit;
  // Favourite index
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  CheckString := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  ExtractString := Trim(ExtractString);
  Val(CheckString, Result.FavouriteIndex, Error);
  if Error <> 0 then
    Exit;
  // Tuning
  ThePos := Pos(' ', ExtractString);
  if ThePos = 0 then
    Exit;
  Result.Tuning := Trim(Copy(ExtractString, 1, ThePos));
  Delete(ExtractString, 1, ThePos);
  // Program name
  Result.ProgramName := ExtractString;
  Result.Active := True;
end;

{------------------------------------------------------------------------------
  Params  : <Definition>  Definition to convert
  Returns : <Result>      String equivalent of definition
                          Empty if error

  Descript: Construct string from information
  Notes   :
 ------------------------------------------------------------------------------}

function ConstructRecordingDefinition(Definition: TRecordItem): ShortString;
var
  Index: Integer;
  Nxt: Boolean;
begin
  Result := '';
  try
    Result := Result + Format('%2.2x', [Definition.RecordId]);
    Result := Result + ' ' + FormatDateTime('YYYYMMDD', Definition.StartTime);
    Result := Result + ' ' + FormatDateTime('HH":"MM":"SS',
      Definition.StartTime);
    Result := Result + ' ' + FormatDateTime('HH":"MM":"SS',
      Definition.StopTime);
    Result := Result + ' [';
    Index := Low(Definition.Pids);
    Nxt := False;
    while (Index <= High(Definition.Pids)) and (Definition.Pids[Index].Pid <>
      $FFFF) do
    begin
      if Nxt then
        Result := Result + ', ';
      Nxt := True;
      Result := Result + Format('%4.4d', [Definition.Pids[Index].Pid]);
      if Definition.Pids[Index].PidName <> '' then
        Result := Result + '(' + Definition.Pids[Index].PidName + ')';
      Inc(Index);
    end;
    Result := Result + ']';
    Result := Result + ' ' + format('%3.3d', [Definition.ListIndex]);
    Result := Result + ' ' + format('%4.4d', [Definition.FavouriteIndex]);
    Result := Result + ' ' + Definition.Tuning;
    Result := Result + ' ' + Definition.ProgramName;
  except
    Result := '';
    Exit;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <ExtractString>   String to extract information from
  Returns : <Result>          Extracted PIDs and optional names

  Descript: Extract PID information from string
            Returned list is in ascending order and no double PIDs exist
  Notes   : PPPP(name), PPPP(name) ...
            PPPP, PPPP, PPPP, ....
            Also allows for less strict definition, for example no comma but
            a space, not fixed number of characters etc.
 ------------------------------------------------------------------------------}

function ExtractFromPidDefinition(ExtractString: ShortString): TPidLists;
var
  PidIndex: Integer;
  CommaPos: Integer;
  Value: Integer;
  Error: Integer;
  NamePos: Integer;
  EndNamePos: Integer;
  CheckPart: ShortString;
  Index1: Integer;
  Index2: Integer;
  TempPid: TPidList;
begin
  // Initial default results
  for Index1 := Low(Result) to High(Result) do
  begin
    Result[Index1].Pid := $FFFF;
    Result[Index1].PidName := '';
  end;

  PidIndex := Low(Result);
  while (ExtractString <> '') and (PidIndex <= High(Result)) do
  begin
    // First try to find (name) and extract it
    // We only do this if we find a '(' before a comma
    NamePos := Pos('(', ExtractString);
    CommaPos := Pos(',', ExtractString);
    if ((NamePos <> 0) and (NamePos < CommaPos)) or
      ((NamePos <> 0) and (CommaPos = 0)) then
    begin
      EndNamePos := Pos(')', ExtractString);
      if EndNamePos = 0 then
      begin
        Result[PidIndex].PidName := Copy(ExtractString, NamePos + 1, 255);
        ExtractString := '';
      end
      else
      begin
        Result[PidIndex].PidName := Copy(ExtractString, NamePos + 1, EndNamePos
          -
          NamePos - 1);
        Delete(ExtractString, NamePos, EndNamePos - NamePos + 1);
      end;
    end;
    ExtractString := Trim(ExtractString);
    // Identifiers are now comma or space
    CommaPos := Pos(',', ExtractString);
    if CommaPos = 0 then
      CommaPos := Pos(' ', ExtractString);
    if CommaPos = 0 then
    begin
      CheckPart := ExtractString;
      ExtractString := '';
    end
    else
    begin
      CheckPart := Copy(ExtractString, 1, CommaPos - 1);
      Delete(ExtractString, 1, CommaPos);
    end;
    Val(CheckPart, Value, Error);
    if (Error = 0) and (Value >= 0) and (Value < 8192) then
    begin
      Result[PidIndex].Pid := Value;
      Inc(PidIndex);
    end;
  end;
  if PidIndex > High(Result) then
    PidIndex := High(Result);
  // Clear all unused PID entries
  if PidIndex < High(Result) then
    for Index1 := PidIndex to High(Result) do
    begin
      Result[Index1].Pid := $FFFF;
      Result[Index1].PidName := '';
    end;
  // We now have all the PIDs from the list. Now strip out any double PIDs if
  // more than a single entry
  if PidIndex > (Low(Result) + 1) then
  begin
    // Mark double entries as invalid
    for Index1 := Low(Result) to PidIndex - 1 do
      for Index2 := Index1 + 1 to PidIndex do
        if Result[Index1].Pid = Result[Index2].Pid then
          Result[Index2].Pid := $FFFF;
    // Move all valid entries to start
    for Index1 := Low(Result) + 1 to PidIndex do
      if (Result[Index1].Pid <> $FFFF) and (Result[Index1 - 1].Pid = $FFFF) then
      begin
        Result[Index1 - 1].Pid := Result[Index1].Pid;
        Result[Index1 - 1].PidName := Result[Index1].PidName;
        Result[Index1].Pid := $FFFF;
        Result[Index1].PidName := '';
      end;
    // Now sort it in ascending order
    for Index1 := Low(Result) to PidIndex do
      for Index2 := Low(Result) to PidIndex - Index1 - 1 do
        if Result[Index2 + 1].Pid < Result[Index2].Pid then
        begin
          TempPid := Result[Index2];
          Result[Index2] := Result[Index2 + 1];
          Result[Index2 + 1] := TempPid;
        end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if a setting has changed

  Descript: Get flag if a setting has changed
  Notes   :
 ------------------------------------------------------------------------------}

function TfrmRecording.GetSettingChanged: Boolean;
var
  Loop: Integer;
  Year: Word;
  Month: Word;
  Day: Word;
  Hour: Word;
  Min: Word;
  Sec: Word;
  Msec: Word;
  StartTime: TDateTime;
  StartDate: TDateTime;
  StopTime: TDateTime;
  CheckDate: TDateTime;
  CheckTime: TDateTime;
  PidIndex: Integer;
  PidList: TPidLists;
begin
  Result := False;
  // Make compiler happy
  CheckTime := Time;
  CheckDate := Date;
  // Check for no definitions at all
  if (sgRecordings.RowCount = 2) and not FSettings[1].Active then
  begin
    if FOrgRowCount <> 0 then
      Result := True;
  end
  else
    Result := Result or (FOrgRowCount <> sgRecordings.RowCount);
  if not Result and (sgRecordings.RowCount > 0) then
    for Loop := 1 to sgRecordings.RowCount - 1 do
    begin
      // Convert original settings
      DecodeDate(FSettings[Loop].StartTime, Year, Month, Day);
      StartDate := EncodeDate(Year, Month, Day);
      DecodeTime(FSettings[Loop].StartTime, Hour, Min, Sec, Msec);
      StartTime := EncodeTime(Hour, Min, Sec, 0);
      DecodeTime(FSettings[Loop].StopTime, Hour, Min, Sec, Msec);
      StopTime := EncodeTime(Hour, Min, Sec, 0);
      // Make compiler happy
      // Check with current settings
      try
        CheckDate := StrToDate(sgRecordings.Cells[3, Loop]);
      except
        Exit;
      end;
      if CheckDate <> StartDate then
        Result := True;
      try
        CheckTime := StrToTime(sgRecordings.Cells[4, Loop]);
      except
        Exit;
      end;
      if CheckTime <> StartTime then
        Result := True;
      try
        CheckTime := StrToTime(sgRecordings.Cells[5, Loop]);
      except
        Exit;
      end;
      if CheckTime <> StopTime then
        Result := True;
      // Get PIDs
      PidList := ExtractFromPidDefinition(sgRecordings.Cells[6, Loop]);
      // Compare them. Since both of them are in ascending order they should
      // match 1:1
      for PidIndex := Low(PidList) to High(PidList) do
        if (FSettings[Loop].Pids[PidIndex].Pid <> PidList[PidIndex].Pid) then
          Result := True;
    end;
end;

{------------------------------------------------------------------------------
  Params  : <CheckData>  Data to check
  Returns : <Result>     Corrected string

  Descript: Convert PID entries into correct string
  Notes   :
 ------------------------------------------------------------------------------}

function TfrmRecording.CheckPids(CheckData: ShortString): ShortString;
var
  Index: Integer;
  TempList: TPidLists;
begin
  Result := '';

  TempList := ExtractFromPidDefinition(CheckData);
  for Index := Low(TempList) to High(TempList) do
    if TempList[Index].Pid <> $FFFF then
      if Result = '' then
        Result := Format('%d', [TempList[Index].Pid])
      else
        Result := Result + Format(', %d', [TempList[Index].Pid]);
  edtPids.Text := Result;
end;

{------------------------------------------------------------------------------
  Params  : <Index>  Item to set
            <Value>  Value for item
  Returns : -

  Descript: Set settings
  Notes   : Format is 'RR < FF > PPPP = DDDDDDDD SS:SS:SS - EE:EE:EE : PPPP(name), PPPP(name) ...., "NAME"
                       1234567890123456789012345678901234567901234567890
                                1         2         3        4         5
 ------------------------------------------------------------------------------}

procedure TfrmRecording.SetSetting(Index: Integer; Value: ShortString);
var
  AllPids: ShortString;
  Check: Integer;
  Loop: Integer;
begin
  if (Index < Low(FOrgSettings)) or (Index > High(FOrgSettings)) then
    Exit;
  if Index >= (sgRecordings.RowCount - 1) then
  begin
    for Loop := sgRecordings.RowCount + 1 to Index + 2 do
    begin
      sgRecordings.RowCount := Loop;
      sgRecordings.Objects[Loop, 0] := nil;
    end;
    FOrgRowCount := sgRecordings.RowCount;
    sgRecordings.FixedRows := 1;
  end;
  FOrgSettings[Index] := ExtractFromRecordingDefinition(Value);
  if not FOrgSettings[Index].Active then
    Exit;
  FSettings[Index + 1] := FOrgSettings[Index];
  sgRecordings.Cells[0, Index + 1] := Format('%2.2x', [Index + 1]);
  sgRecordings.Cells[1, Index + 1] := FSettings[Index + 1].ProgramName;
  if FSettings[Index + 1].RecordId <> $FF then
    sgRecordings.Cells[2, Index + 1] := 'Yes'
  else
    sgRecordings.Cells[2, Index + 1] := 'No';
  sgRecordings.Cells[3, Index + 1] := DateToStr(FSettings[Index + 1].StartTime);
  sgRecordings.Cells[4, Index + 1] := TimeToStr(FSettings[Index + 1].StartTime);
  sgRecordings.Cells[5, Index + 1] := TimeToStr(FSettings[Index + 1].StopTime);
  AllPids := '';
  Check := Low(FSettings[Index + 1].Pids);
  while (Check <= High(FSettings[Index + 1].Pids)) and (FSettings[Index +
    1].Pids[Check].Pid <> $FFFF) do
  begin
    if AllPids = '' then
      AllPids := Format('%d', [FSettings[Index + 1].Pids[Check].Pid])
    else
      AllPids := AllPids + Format(', %d', [FSettings[Index +
        1].Pids[Check].Pid]);
    Inc(Check);
  end;
  sgRecordings.Cells[6, Index + 1] := AllPids;
  sgRecordingsClick(nil);
end;

{------------------------------------------------------------------------------
  Params  : <Index>  Index of setting
  Returns : <Result> Setting on indicated <Index>. Empty if not available or
                     error.

  Descript: Get setting
  Notes   :
 ------------------------------------------------------------------------------}

function TfrmRecording.GetSetting(Index: Integer): ShortString;
var
  RecordItem: TRecordItem;
  Loop: Integer;
  Loop2: Integer;
begin
  Result := '';
  if (Index < Low(FOrgSettings)) or (Index > High(FOrgSettings)) then
    Exit;
  if (Index >= (sgRecordings.RowCount - 1)) then
    Exit;
  if not FSettings[Index + 1].Active then
    Exit;

  // Convert original settings
  RecordItem.RecordId := FSettings[Index + 1].RecordId;
  RecordItem.ProgramName := FSettings[Index + 1].ProgramName;
  RecordItem.ListIndex := FSettings[Index + 1].ListIndex;  // V0.061 bugfix
  RecordItem.FavouriteIndex := FSettings[Index + 1].FavouriteIndex; // V0.061 bugfix
  RecordItem.Tuning := FSettings[Index + 1].Tuning;
  // Add user settings
  try
    RecordItem.StartTime := StrToDate(sgRecordings.Cells[3, Index + 1]);
  except
    Exit;
  end;
  try
    // Initially use same date
    RecordItem.StopTime := RecordItem.StartTime +
      StrToTime(sgRecordings.Cells[5, Index + 1]);
  except
    Exit;
  end;
  try
    RecordItem.StartTime := RecordItem.StartTime +
      StrToTime(sgRecordings.Cells[4, Index + 1]);
  except
    Exit;
  end;
  // Correct for a single day
  if RecordItem.StopTime < RecordItem.StartTime then
    RecordItem.StopTime := RecordItem.StopTime + 1;
  // Get PIDs
  RecordItem.Pids := ExtractFromPidDefinition(sgRecordings.Cells[6, Index + 1]);
  // Find names of the PIDs
  for Loop := Low(RecordItem.Pids) to High(RecordItem.Pids) do
    for Loop2 := Low(FSettings[Index + 1].Pids) to High(FSettings[Index +
      1].Pids) do
      if RecordItem.Pids[Loop].Pid = FSettings[Index + 1].Pids[Loop2].Pid then
        RecordItem.Pids[Loop].PidName := FSettings[Index +
          1].Pids[Loop2].PidName;
  Result := ConstructRecordingDefinition(RecordItem);
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Revert to orginal settings
  Notes   :
 ------------------------------------------------------------------------------}

procedure TfrmRecording.Revert;
var
  Index: Integer;
  AllPids: ShortString;
  Check: Integer;
begin
  sgRecordings.RowCount := FOrgRowCount;
  for Index := 1 to sgRecordings.RowCount - 1 do
  begin
    FSettings[Index] := FOrgSettings[Index - 1];
    sgRecordings.Cells[0, Index] := format('%2.2d', [Index]);
    sgRecordings.Cells[1, Index] := FSettings[Index].ProgramName;
    if FSettings[Index].RecordId <> $FF then
      sgRecordings.Cells[2, Index] := 'Yes'
    else
      sgRecordings.Cells[2, Index] := 'No';
    sgRecordings.Cells[3, Index] := DateToStr(FSettings[Index].StartTime);
    sgRecordings.Cells[4, Index] := TimeToStr(FSettings[Index].StartTime);
    sgRecordings.Cells[5, Index] := TimeToStr(FSettings[Index].StopTime);
    AllPids := '';
    Check := Low(FSettings[Index].Pids);
    while (Check <= High(FSettings[Index].Pids)) and
      (FSettings[Index].Pids[Check].Pid <> $FFFF) do
    begin
      if AllPids = '' then
        AllPids := Format('%d', [FSettings[Index].Pids[Check].Pid])
      else
        AllPids := AllPids + Format(', %d', [FSettings[Index].Pids[Check].Pid]);
      Inc(Check);
    end;
    sgRecordings.Cells[6, Index] := AllPids;
  end;
end;

procedure TfrmRecording.btnRemoveClick(Sender: TObject);
var
  Loop: Integer;
  Loop2: Integer;
begin
  if sgRecordings.RowCount <= 1 then
    Exit;
  if sgRecordings.RowCount > 2 then
  begin
    // Copy items below removed item
    for Loop := sgRecordings.Row + 1 to sgRecordings.RowCount - 1 do
    begin
      FSettings[Loop - 1] := FSettings[Loop];
      for Loop2 := 1 to sgRecordings.ColCount - 1 do
        sgRecordings.Cells[Loop2, Loop - 1] := sgRecordings.Cells[Loop2, Loop];
    end;
    sgRecordings.RowCount := sgRecordings.RowCount - 1;
    if sgRecordings.Row > 1 then
      sgRecordings.Row := sgRecordings.Row - 1;
  end
  else
  begin
    sgRecordings.Row := 1;
    sgRecordings.RowCount := 2;
    FSettings[1].Active := False;
    sgRecordings.Cells[1, 1] := '';
    sgRecordings.Cells[2, 1] := '';
    sgRecordings.Cells[3, 1] := '';
    sgRecordings.Cells[4, 1] := '';
    sgRecordings.Cells[5, 1] := '';
    sgRecordings.Cells[6, 1] := '';
  end;
end;

procedure TfrmRecording.btnRemoveAllClick(Sender: TObject);
begin
  if sgRecordings.RowCount <= 1 then
    Exit;
  while sgRecordings.RowCount > 2 do
    btnRemoveClick(Sender);
  btnRemoveClick(Sender);
end;

procedure TfrmRecording.btnUpClick(Sender: TObject);
var
  TempString: string;
  Column: Integer;
  RecordItem: TRecordItem;
begin
  if sgRecordings.Row < 2 then
    Exit;
  RecordItem := FSettings[sgRecordings.Row - 1];
  FSettings[sgRecordings.Row - 1] := FSettings[sgRecordings.Row];
  FSettings[sgRecordings.Row] := RecordItem;
  for Column := 1 to 7 do
  begin
    TempString := sgRecordings.Cells[Column, sgRecordings.Row - 1];
    sgRecordings.Cells[Column, sgRecordings.Row - 1] :=
      sgRecordings.Cells[Column, sgRecordings.Row];
    sgRecordings.Cells[Column, sgRecordings.Row] := TempString;
  end;
  sgRecordings.Row := sgRecordings.Row - 1;
end;

procedure TfrmRecording.btnDownClick(Sender: TObject);
var
  TempString: string;
  Column: Integer;
  RecordItem: TRecordItem;
begin
  if sgRecordings.Row >= (sgRecordings.RowCount - 1) then
    Exit;
  RecordItem := FSettings[sgRecordings.Row + 1];
  FSettings[sgRecordings.Row + 1] := FSettings[sgRecordings.Row];
  FSettings[sgRecordings.Row] := RecordItem;
  for Column := 1 to 7 do
  begin
    TempString := sgRecordings.Cells[Column, sgRecordings.Row + 1];
    sgRecordings.Cells[Column, sgRecordings.Row + 1] :=
      sgRecordings.Cells[Column, sgRecordings.Row];
    sgRecordings.Cells[Column, sgRecordings.Row] := TempString;
  end;
  sgRecordings.Row := sgRecordings.Row + 1;
end;

procedure TfrmRecording.sgRecordingsClick(Sender: TObject);
begin
  sgRecordings.Col := 0;
  dtStartRecordingDate.Date := StrToDate(sgRecordings.Cells[3,
    sgRecordings.Row]);
  dtStartRecording.Time := StrToTime(sgRecordings.Cells[4, sgRecordings.Row]);
  dtStopRecording.Time := StrToTime(sgRecordings.Cells[5, sgRecordings.Row]);
  edtPids.Text := sgRecordings.Cells[6, sgRecordings.Row];
end;

procedure TfrmRecording.edtPidsKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
  begin
    sgRecordings.Cells[6, sgRecordings.Row] := CheckPids(edtPids.Text);
    edtPids.SelStart := 255;
  end;
end;

procedure TfrmRecording.edtPidsExit(Sender: TObject);
begin
  sgRecordings.Cells[6, sgRecordings.Row] := CheckPids(edtPids.Text);
  edtPids.SelStart := 255;
end;

procedure TfrmRecording.dtStartRecordingDateChange(Sender: TObject);
begin
  sgRecordings.Cells[3, sgRecordings.Row] :=
    DateToStr(dtStartRecordingDate.Date);
end;

procedure TfrmRecording.dtStartRecordingChange(Sender: TObject);
begin
  sgRecordings.Cells[4, sgRecordings.Row] := TimeToStr(dtStartRecording.Time);
end;

procedure TfrmRecording.dtStopRecordingChange(Sender: TObject);
begin
  sgRecordings.Cells[5, sgRecordings.Row] := TimeToStr(dtStopRecording.Time);
end;

procedure TfrmRecording.btnRevertClick(Sender: TObject);
begin
  Revert;
end;

procedure TfrmRecording.btnOKClick(Sender: TObject);
begin
  FAccepted := True;
  Hide;
  Refresh;
  if Assigned(@FExitCallback) then
    FExitCallback(nil);
  // Never call it again
  @FExitCallback := nil;
  Close;
end;

procedure TfrmRecording.btnDiscardClick(Sender: TObject);
begin
  Hide;
  Refresh;
  Revert;
  if Assigned(@FExitCallback) then
    FExitCallback(nil);
  // Never call it again
  @FExitCallback := nil;
  Close;
end;

procedure TfrmRecording.FormCreate(Sender: TObject);
var
  Loop: Integer;
begin
  @FExitCallback := nil;
  FAccepted := False;
  for Loop := Low(FOrgSettings) to High(FOrgSettings) do
    FOrgSettings[Loop].Active := True;
  FOrgRowCount := 0;
  sgRecordings.ColWidths[0] := 20;
  sgRecordings.ColWidths[1] := 100;
  sgRecordings.ColWidths[2] := 30;
  sgRecordings.ColWidths[3] := 60;
  sgRecordings.ColWidths[4] := 50;
  sgRecordings.ColWidths[5] := 50;
  sgRecordings.ColWidths[6] := 150;
  sgRecordings.Cells[1, 0] := 'Program';
  sgRecordings.Cells[2, 0] := 'Rec';
  sgRecordings.Cells[3, 0] := 'Date';
  sgRecordings.Cells[4, 0] := 'Start';
  sgRecordings.Cells[5, 0] := 'Stop';
  sgRecordings.Cells[6, 0] := 'PIDs';
  dtStartRecordingDate.Date := Date;
end;

procedure TfrmRecording.FormDestroy(Sender: TObject);
begin
  btnDiscardClick(nil);
  @FExitCallback := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}

procedure Initialize;
begin
  //
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}

procedure Finalize;
begin
  //
end;

initialization
  Initialize;

finalization
  Finalize;

end.

