{****************************************************************************}
{ Program: RemoteGenerics.pas                                                }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ Generic remote control functions/procedures/constants                      }
{                                                                            }
{ Version                                                                    }
{  1.00   20020916  - First release                                          }
{****************************************************************************}
unit RemoteGenerics;

interface

uses
  Windows;

const
  // Index constants Pronto data
  CPFIdentifier = 0;
  CPFFrequency  = 1;
  CPFOnce       = 2;
  CPFRepeat     = 3;
  CPFFirstPair  = 4;

  CPFModulated   = $0000;
  CPFUnmodulated = $0100;

  CPFUnit          = 0.241246;
  CPFUnitReciproke = 1/CPFUnit;

type
  PDWordArray = ^TDWordArray;
  TDWordArray = array[0..32767] of Dword;
  PWordArray = ^TWordArray;
  TWordArray = array[0..32767] of Word;

  function EncodeBits(Data: Word; FromBit: Byte; ToBit: Byte; AZero: AnsiString; AOne: AnsiString): AnsiString;
  function ZeroOneSequences(BitStream: AnsiString; Delay: Word; Carrier: Dword; BitTime: Dword; UseRepeat: boolean): AnsiString;
  function ASCIItoWordArray(InputString: AnsiString; Buffer: PWordArray): Word;
  function ASCIItoDwordArray(InputString: AnsiString; Buffer: PDwordArray): Word;
  function WordArrayToASCII(Buffer: PWordArray): AnsiString;
  function DwordArrayToASCII(Buffer: PDwordArray): AnsiString;
  function DetectCarrierFromDwordArray(Buffer: PDwordArray): Longint;
  function UnmodulatedToModulatedFormat(Unmodulated: AnsiString): AnsiString;
  function AppendASCIIFormatted(First, Second: AnsiString): AnsiString;

implementation

uses
  SysUtils;


{------------------------------------------------------------------------------
  Params   : <InputString>  ASCII data to convert
             <Buffer>       Pointer to buffer
  Return   : <Result>       Size of array data (PRONTO size!)

  Descript : Convert ASCII data into dword array data
  Notes    : -
 ------------------------------------------------------------------------------}
function ASCIItoDwordArray(InputString: AnsiString; Buffer: PDwordArray): Word;
var
  Part : string;
  Value: Dword;                                  // Conversion value
  Error: Integer;                                // Error indicator
  Index: Word;
begin
  // First convert some special codes (CR/LF/TAB) into spaces }
  Index  := 0;
  while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
  while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
  while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';
  while (length(InputString)<>0) do
  begin
    Part := '';
    // Remove leading spaces
    while (length(InputString)<>0) and (InputString[1]=' ') do
      Delete(InputString, 1, 1);
    // Get all characters until a space or end of the string
    while (length(InputString)<>0) and (InputString[1]<>' ') do
    begin
      Part := Part + InputString[1];
      Delete(InputString, 1, 1);
    end;
    // Convert the value
    if Part<>'' then
    begin
      Part := '$' + Part;
      Val(Part, Value, Error);
      if Error <> 0 then
      begin
        Result := 0;
        Exit;
      end
      else
      begin
        Buffer[Index] := Value;
        Inc(Index);
        if Index > High(TDwordArray) then
        begin
          Result := 0;
          Exit;
        end;
      end;
    end;
  end;
  Result := (Index - CPFFirstPair) div 2;
end;


{------------------------------------------------------------------------------
  Params   : <InputString>  ASCII data to convert
             <Buffer>       Pointer to buffer
  Return   : <Result>       Size of array data (PRONTO size!)

  Descript : Convert ASCII data into dword array data
  Notes    : -
 ------------------------------------------------------------------------------}
function ASCIItoWordArray(InputString: AnsiString; Buffer: PwordArray): Word;
var
  Part : string;
  Value: Word;                                  // Conversion value
  Error: Integer;                                // Error indicator
  Index: Word;
begin
  // First convert some special codes (CR/LF/TAB) into spaces }
  Index  := 0;
  while (Pos(#8 , InputString) <> 0) do InputString[Pos(#8 , InputString)] := ' ';
  while (Pos(#10, InputString) <> 0) do InputString[Pos(#10, InputString)] := ' ';
  while (Pos(#13, InputString) <> 0) do InputString[Pos(#13, InputString)] := ' ';
  while (length(InputString)<>0) do
  begin
    Part := '';
    // Remove leading spaces
    while (length(InputString)<>0) and (InputString[1]=' ') do
      Delete(InputString, 1, 1);
    // Get all characters until a space or end of the string
    while (length(InputString)<>0) and (InputString[1]<>' ') do
    begin
      Part := Part + InputString[1];
      Delete(InputString, 1, 1);
    end;
    // Convert the value
    if Part<>'' then
    begin
      Part := '$' + Part;
      Val(Part, Value, Error);
      if Error <> 0 then
      begin
        Result := 0;
        Exit;
      end
      else
      begin
        Buffer[Index] := Value;
        Inc(Index);
        if Index > High(TDwordArray) then
        begin
          Result := 0;
          Exit;
        end;
      end;
    end;
  end;
  Result := (Index - CPFFirstPair) div 2;
end;


{------------------------------------------------------------------------------
  Params   : <Buffer>  Pointer to buffer (MUST contain Pronto size!)
  Return   : <Result>  Result string
                       Empty if invalid

  Descript : Convert data input to string representation.
  Notes    : Since we also have to take values > $FFFF into account we will
             translate these to more than 4 digits ....
             This happens only with our 'special' recording format...
 ------------------------------------------------------------------------------}
function DwordArrayToASCII(Buffer: PDwordArray): AnsiString;
var
  AString: AnsiString;
  Loop   : Word;
  Index  : Word;
begin
  Index := ((Buffer[CPFOnce] + Buffer[CPFRepeat]) + 2) * 2;
  if (Index > 4) and
     (Index < High(TDwordArray)) then
  begin
    AString := '';
    for Loop := 0 to Index-1 do
      if Loop = 0 then
      begin
        if Buffer[Loop] < $10000 then
          AString := AString + format('%.4x', [Buffer[Loop]])
        else
          AString := AString + format('%.8x', [Buffer[Loop]]);
      end
      else
      begin
        if Buffer[Loop] < $10000 then
          AString := AString + format(' %.4x', [Buffer[Loop]])
        else
          AString := AString + format(' %.8x', [Buffer[Loop]]);
      end;
    Result := AString;
  end
  else
    Result := '';
end;


{------------------------------------------------------------------------------
  Params   : <Buffer>  Pointer to buffer (MUST contain Pronto size!)
  Return   : <Result>  Result string
                       Empty if invalid

  Descript : Convert data input to string representation.
  Notes    : Since we also have to take values > $FFFF into account we will
             translate these to more than 4 digits ....
             This happens only with our 'special' recording format...
 ------------------------------------------------------------------------------}
function WordArrayToASCII(Buffer: PWordArray): AnsiString;
var
  AString: AnsiString;
  Loop   : Word;
  Index  : Word;
begin
  Index := ((Buffer[CPFOnce] + Buffer[CPFRepeat]) + 2) * 2;
  if (Index > 4) and
     (Index < High(TWordArray)) then
  begin
    AString := '';
    for Loop := 0 to Index-1 do
      if Loop = 0 then
      begin
        AString := AString + format('%.4x', [Buffer[Loop]])
      end
      else
      begin
        AString := AString + format(' %.4x', [Buffer[Loop]])
      end;
    Result := AString;
  end
  else
    Result := '';
end;


{------------------------------------------------------------------------------
  Params  : <Data>       Data bit s to convert
            <FromBit>    Starting bit [0..15]
            <ToBit>      Ending bit to convert [0..15]
            <AZero>      How a '0' bit is to be converted to
            <AOne>       How a '1' bit is to be converted to
  Returns : <Result>     Result string

  Descript: Encode bits to a string representation
  Notes   : -
 ------------------------------------------------------------------------------}
function EncodeBits(Data: Word; FromBit: Byte; ToBit: Byte; AZero: Ansistring; AOne: AnsiString): AnsiString;
var
  BitString: AnsiString;
  Loop     : integer;
begin
  BitString := '';
  if not FromBit in [0..15] then Exit;
  if not ToBit   in [0..15] then Exit;

  if ToBit < FromBit then
  begin
    for Loop := FromBit downto ToBit do
    begin
      if (Data and (1 shl Loop)) = 0 then
        BitString := BitString + AZero
      else
        BitString := BitString + AOne;
    end;
  end
  else
  begin
    for Loop := FromBit to ToBit do
    begin
      if (Data and (1 shl Loop)) = 0 then
        BitString := BitString + AZero
      else
        BitString := BitString + AOne;
    end;
  end;
  Result := BitString;
end;


{------------------------------------------------------------------------------
  Params  : <BitStream>      Bitstream string to convert (made up of ASCII '1'/'0')
            <Delay>          Delay in ms (delay ending the code)
            <Carrier>        Carrier frequency in Hz
            <BitTime>        Bittime in us
            <UseRepeat>      Generate 'repeat' result
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Convert a bitstream into 1<>0 sequences (PRONTO format)
  Notes   : -
 ------------------------------------------------------------------------------}
function ZeroOneSequences(BitStream: AnsiString; Delay: Word; Carrier: Dword; BitTime: Dword; UseRepeat: boolean): AnsiString;
var
  ProcessString: string;
  RCResult     : string;
  Count1       : integer;
  Count0       : integer;
  DelayUnit    : integer;
  Items        : integer;
  Frequency    : word;
  Frequency2   : word;
  CarrierUnit  : word;
  TimingUnit   : word;
begin
  ProcessString := BitStream;
  Items         := 0;

  // First the timing info to use
  if Carrier = 0 then
    Exit
  else
    Frequency := Carrier;
  CarrierUnit := round(1E6/(Frequency  * 0.241246));       // Actual timing unit
  Frequency2  := trunc(1E6/(CarrierUnit * 0.241246));      // Actual frequency
  TimingUnit  := round(BitTime/1E6 * Frequency2);

  // First we have some minor cleaning up to do. We must make sure we start
  // with a '1' and end with a '1'. A leading '0' can be removed and a
  // trailing '0' might be added.
  // Because we do this our string is always correct for the next stages.
  if (length(ProcessString)<>0) and (ProcessString[1]='0') then
    Delete(ProcessString, 1, 1);
  if (length(ProcessString)<>0) and (ProcessString[length(ProcessString)]='1') then
    ProcessString := ProcessString + '0';
  // Now count them and while we are at it translate it into the result:
  // Each 1->0 sequence is translated into two (4-digit hexadecimal) numbers.
  // These two numbers indicate the ON time (1) and the OFF time (0) of the
  // sequence. It is in units of the carrier frequency. We already calculated/
  // defined the TimingUnits required for each bit in the string.
  // For the delay time we need to calculate it seperately.
  DelayUnit := round((Delay / 1000) * Frequency2);         // Delay in ms
  if DelayUnit < TimingUnit then
    DelayUnit := TimingUnit;
  while length(ProcessString)<>0 do
  begin
    inc(Items);
    Count0 := 0;
    Count1 := 0;
    while (length(ProcessString)<>0) and (ProcessString[1]='1') do   // The added security (length check) is not really necessary
    begin
      inc(Count1);
      delete(ProcessString, 1, 1);
    end;
    while (length(ProcessString)<>0) and (ProcessString[1]='0') do   // The added security (length check) is not really necessary
    begin
      inc(Count0);
      delete(ProcessString, 1, 1);
    end;
    // We now have counted the number of ones and zeroes in one sequence, now
    // convert them into hexadecimal numbers.
    // A special case is if all data is processed; then we need to use the
    // requested delay.
    if length(ProcessString)<>0 then
      RCResult := RCResult + format(' %.4x %.4x', [Count1 * TimingUnit, Count0 * TimingUnit])
    else
      RCResult := RCResult + format(' %.4x %.4x', [Count1 * TimingUnit, DelayUnit]);
  end;
  // Now all we have to do is remove the single leading space
  if (length(RCResult)<>0) and (RCResult[1]=' ') then
    Delete(RCResult, 1, 1);

  // Now we have all our codes in a single string we only need to add the
  // header:
  // 0000 CCCC IIII 0000
  // CCCC = carrier frequency      Frequency = 1000000/(N * .241246)
  //                               1000000 / (Frequency * 0.241246)
  // IIII = total items
  //                         OR FOR REPEAT
  // 0000 CCCC 0000 IIII
  // CCCC = carrier frequency      Frequency = 1000000/(N * .241246)
  //                               1000000 / (Frequency * 0.241246)
  // IIII = total items
  // Again all values are 4-digits hexadecimal
  if UseRepeat then
    RCResult := '0000 ' + format('%.4x ', [CarrierUnit]) +
                format('0000 %.4x ', [Items]) + RCResult
  else
    RCResult := '0000 ' + format('%.4x ', [CarrierUnit]) +
                format('%.4x 0000 ', [Items]) + RCResult;

  Result := RCResult;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>  Pointer to dword data
  Returns : <Result>  Carrier frequency in Hz
                      0 if an error was detected

  Descript: Scan data for carrier data.
  Notes   : -
 ------------------------------------------------------------------------------}
function DetectCarrierFromDwordArray(Buffer: PDwordArray): Longint;
var
  Loop           : Word;
  Calc           : Extended;
  ActiveSample   : Longint;
  InActiveSample : Longint;
  Average        : Extended;
  Averaged       : Word;
begin
  Result := 0;
  // Sanity checks
  if Buffer[CPFOnce] = 0 then
    Exit;
  // Go through all the data to detect carrier frequency
  // Depending on the source we have to go through the data differently
  Loop     := CPFFirstPair;                      // Skip header
  Average  := 0;
  Averaged := 0;
  repeat
    ActiveSample   := Buffer[Loop];
    inc(Loop);
    InActiveSample := Buffer[Loop];
    inc(Loop);
    // Only takes samples which are no more than 1::4
    if ((ActiveSample <> 0) and (InActiveSample <> 0)) then
      if ((ActiveSample <= InActiveSample) and ((ActiveSample   / InActiveSample) > 0.25)) or
         ((ActiveSample >= InActiveSample) and ((InActiveSample / ActiveSample  ) > 0.25)) then
      begin
        Average := Average + ActiveSample + InActiveSample;
        inc(Averaged);
      end;
  until Loop >= (CPFFirstPair + (Buffer[CPFOnce] * 2));
  if Averaged <> 0 then
  begin
    Calc := Average / Averaged;
    Calc := Calc * Buffer[CPFFrequency];
    // The type of data can also indicate the multiplier applied for the timing
    if (Buffer[CPFIdentifier] <> CPFModulated) and
       (Buffer[CPFIdentifier] <> CPFUnmodulated) then
       Calc := Calc / Buffer[CPFIdentifier];
    Calc := 1E6 / (Calc * CPFUnit);
    Result := Round(Calc);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Recorded>   Recorded data
  Returns : <Result>     First ASCII value indicates the success
                            0001     Suspicious timing
                            0000     No obvious errors
                            FFFF     No data
                            FFFE     Invalid data
                            FFFC     No carrier detected
                            FFFB     Invalid number of data
                            FFFA     On/Off time, not carrier based signal
                            FFF9     Too much data
                            FFF8     Too much deviation from carrier signal
                            FFF6     Discrepency in data

  Descript: Convert recorded unmodulated format to modulated Pronto format
            The direct data recording is changed into Pronto format.
            This takes the following into account:
            . the carrier frequency is determined
            . an ON signal (which is made up of the carrier frequency) is
              determined -> note that the carrier signal has most likely no
              equal on/off timing (which would be the theoretical input
            . ON and OFF signals are determined (carrier or not)
            The recorded format uses the unmodulated ('0100') Pronto format.
            When no '0100' is detected it is assumes this is the divider
            for the timing (frequency). This has been implemented to increase
            the resolution of the timing/frequency. Example: when the format
            indicates '03E8' (=1000) then the timing information needs to be
            divided by 1000 to get the actual value.
  Notes   : -
 ------------------------------------------------------------------------------}
function UnmodulatedToModulatedFormat(Unmodulated: AnsiString): AnsiString;
var
  Carrier      : Longint;
  Calc         : Extended;
  SourceTimeUs : Extended;
  TargetTimeUs : Extended;
  Loop         : Integer;
  OnTime       : Extended;
  OffTime      : Extended;
  CarrierCount : Integer;
  TooShort     : Boolean;
  Buffer       : TDwordArray;
  ChkSize      : Word;
  Size         : Integer;
begin
  Result := '';
  TooShort := False;
  // Do some sanity checks
  ChkSize := ASCIIToDwordArray(Unmodulated, @Buffer);
  if ChkSize = 0 then
  begin
    Result := 'FFFF';
    Exit;
  end;
  if ChkSize < 6 then
  begin
    Result := 'FFFE';
    Exit;
  end;
  Carrier := DetectCarrierFromDwordArray(@Buffer);
  if Carrier = 0 then
  begin
    Result := 'FFF7';
    Exit;
  end;
  if Buffer[CPFOnce] = 0 then
    Size := Buffer[CPFRepeat]
  else
    Size := Buffer[CPFOnce];
  if ChkSize <> Size then
  begin
    Result := 'FFF6';
    Exit;
  end;

  // The carrier is usually a number rounded to 1000 Hz...
  Carrier := ((Carrier + 500) div 1000) * 1000;

  // To recap; the Pronto format is made up of a header and the data.
  //  The header:
  //    0000 CCCC IIII 0000
  //    . CCCC = carrier frequency      Frequency = 1000000/(N * .241246)
  //                                    N = 1000000 / (Frequency * 0.241246)
  //    . IIII = total number of data numbers following
  SourceTimeUs := Buffer[CPFFrequency] * CPFUnit;
  if (Buffer[CPFIdentifier] <> CPFModulated) and
     (Buffer[CPFIdentifier] <> CPFUnmodulated) then
    SourceTimeUs := SourceTimeUs / Buffer[CPFIdentifier];

  // Convert detected carrier into Pronto value
  Calc := 1E6 / (Carrier * CPFUnit);
  Buffer[CPFFrequency] := Round(Calc);
  Size                 := CPFFirstPair;          // No pairs detected yet
  TargetTimeUs := (1/Carrier)/2 * 1E6;           // Factor /2 is because 1 frequency cycle = on + off

  CarrierCount := 0;
  Loop := CPFFirstPair;
  repeat
    OnTime := Buffer[Loop];
    inc(Loop);
    OffTime := Buffer[Loop];
    inc(Loop);
    OnTime  := OnTime  * SourceTimeUs;
    OffTime := OffTime * SourceTimeUs;
    if OnTime > (OffTime * 4) then
    begin
      Result := 'FFFA';
      Exit;
    end;
    // A long off-time follows (end of carrier)
    if OffTime > (OnTime * 4) then
    begin
      inc(CarrierCount);
      Buffer[Size] := CarrierCount;
      if Buffer[Size] <= 4 then       // If less than 5 carrier cycles detected, probably error
        TooShort := True;
      inc(Size);
      if Size > High(TDwordArray) then
      begin
        Result := 'FFF9';
        Exit;
      end;
      // Convert off-time to equivalent of carrier cycles (=on+off)
      Buffer[Size] := round((OffTime - OnTime) / (TargetTimeUs * 2));
      if Buffer[Size] <= 4 then       // If less than 5 carrier cycles detected, probbaly error
        TooShort := True;
      inc(Size);
      if Size > High(TDwordArray) then
      begin
        Result := 'FFF9';
        Exit;
      end;
      CarrierCount := 0;
    end
    else                                         // If carrier signal
    begin
      if (OnTime + OffTime) > ((TargetTimeUs * 2) * 4 {1.5}) then
      begin
        Result := 'FFF8';
        Exit;
      end
      else inc(CarrierCount);
    end;
  until (Loop >= ((ChkSize * 2) + CPFFirstPair));
  // Save pairs
  if Buffer[CPFOnce] = 0 then
    Buffer[CPFRepeat] := (Size - CPFFirstPair) div 2
  else
    Buffer[CPFOnce] := (Size - CPFFirstpair) div 2;
  if TooShort then
    Buffer[CPFIdentifier] := CPFModulated + 1
  else
    Buffer[CPFIdentifier] := CPFModulated;
  Result := DwordArrayToASCII(@Buffer);
end;


{------------------------------------------------------------------------------
  Params  : <First>   First Pronto data (reference)
            <Second>  Second Pronto data
  Returns : -

  Descript: Append Pronto formatted data (ASCII)
  Notes   : Data (carrier and such) from the second data is discarded
 ------------------------------------------------------------------------------}
function AppendASCIIFormatted(First, Second: AnsiString): AnsiString;
var
  TargetBuffer  : TDwordArray;
  FirstBuffer   : TDwordArray;
  SecondBuffer  : TDwordArray;
  FirstChkSize  : Word;
  SecondChkSize : Word;
  Index         : Word;
  TargetIndex   : Word;
begin
  Result := '';
  FirstChkSize  := ASCIIToDwordArray(First, @FirstBuffer);
  SecondChkSize := ASCIIToDwordArray(Second, @SecondBuffer);
  if FirstChkSize = 0 then
  begin
    Result := Second;
    Exit;
  end;
  if SecondChkSize = 0 then
  begin
    Result := First;
    Exit;
  end;
  TargetBuffer[CPFIdentifier] := FirstBuffer[CPFIdentifier];
  TargetBuffer[CPFFrequency]  := FirstBuffer[CPFFrequency];
  TargetIndex := CPFFirstPair;

  if FirstBuffer[CPFOnce] <> 0 then
    for Index := 0 to (FirstBuffer[CPFOnce]*2)-1 do
    begin
      TargetBuffer[TargetIndex] := FirstBuffer[CPFFirstPair + Index];
      Inc(TargetIndex);
    end;
  if SecondBuffer[CPFOnce] <> 0 then
    for Index := 0 to (SecondBuffer[CPFOnce]*2)-1 do
    begin
      TargetBuffer[TargetIndex] := SecondBuffer[CPFFirstPair + Index];
      Inc(TargetIndex);
    end;
  if TargetIndex <> CPFFirstpair then
    TargetBuffer[CPFOnce] := (TargetIndex div 2) - 2
  else
    TargetBuffer[CPFOnce] := 0;

  if FirstBuffer[CPFRepeat] <> 0 then
    for Index := 0 to (FirstBuffer[CPFRepeat]*2)-1 do
    begin
      TargetBuffer[TargetIndex] := FirstBuffer[CPFFirstPair + Index + (FirstBuffer[CPFOnce]*2)];
      Inc(TargetIndex);
    end;
  if SecondBuffer[CPFRepeat] <> 0 then
    for Index := 0 to (SecondBuffer[CPFRepeat]*2)-1 do
    begin
      TargetBuffer[TargetIndex] := SecondBuffer[CPFFirstPair + Index + (SecondBuffer[CPFOnce]*2)];
      Inc(TargetIndex);
    end;
  if TargetIndex <> CPFFirstpair then
    TargetBuffer[CPFRepeat] := ((TargetIndex div 2) - 2) - TargetBuffer[CPFOnce]
  else
    TargetBuffer[CPFRepeat] := 0;

  Result := DWordArrayToASCII(@TargetBuffer);
end;

end.
