{****************************************************************************}
{ Program: RC6a.pas                                                          }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ RC6a functions/procedures                                                  }
{                                                                            }
{ Version                                                                    }
{  1.00   20020901  - First release                                          }
{****************************************************************************}
unit RC6a;

interface

uses
  Windows;

{$I PRONTORCX.INC}

  function GenerateCodeRC6A(Parameters: TRCParameters): AnsiString;

  function HalfbitBitstreamToProntoCodeRC6A(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
  function ParametersToBitstreamRC6A(Parameters: TRCParameters): AnsiString;
  function BitstreamToHalfbitBitstreamRC6A(Bitstream: AnsiString): AnsiString;

implementation

uses
  RemoteGenerics;

const
  // Settings for generating RC5/RC5X/RC6/RC6A code
  CRCFrequency   = 36000;              // Default carrier frequency
  CRCBitTime     = 889 div 2;          // Bit time in us
  CRCDelay       = 69;                 // Default 69 ms delay after codes


{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single bitstream code from the parameters
  Notes   : -
 ------------------------------------------------------------------------------}
function ParametersToBitstreamRC6A(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
  Zero    : char;
  One     : char;
begin
  Result    := '';
  if Parameters[rptAddress] > 255  then Exit;
  if Parameters[rptCommand] > 255  then Exit;
  if (Parameters[rptCustomer] > 127) and
     (Parameters[rptCustomer] < 32768) then Exit;

  RCString := '';
  Zero := '0';
  One  := '1';

  RCString := '000000000';
  if Parameters[rptToggle]<>0 then
    RCString := RCString + '10'
  else
    RCString := RCString + '01';
  if Parameters[rptCustomer] < 128 then
  begin
    RCString := RCString + '0';
    RCString := RCString + EncodeBits(Parameters[rptCustomer], 6, 0, Zero, One);
  end
  else
  begin
    RCString := RCString + '1';
    RCString := RCString + EncodeBits(Parameters[rptCustomer], 14, 0, Zero, One);
  end;

  RCString := RCString + EncodeBits(Parameters[rptAddress], 7, 0, Zero, One);
  RCString := RCString + EncodeBits(Parameters[rptCommand], 7, 0, Zero, One);

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <Bitstream>      Bitstream to convert
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Convert a bitstream into an (halfbit) encoded bitstream (bi-phase encoded)
  Notes   : -
 ------------------------------------------------------------------------------}
function BitstreamToHalfbitBitstreamRC6A(Bitstream: AnsiString): AnsiString;
var
  RCString   : AnsiString;
  Zero       : string[2];
  One        : string[2];
  BiphaseZero: string[2];
  BiphaseOne : string[2];
  Index      : Integer;
begin
  Result    := '';
  // Sanity checks on data and length
  for Index := 1 to Length(Bitstream) do
    if not (BitStream[Index] in ['0', '1']) then
      Exit;
  if (Length(Bitstream) <> 35) and
     (Length(Bitstream) <> 43) then Exit;

  if (Bitstream[12] = '0') and
     (Length(Bitstream) <> 35) then Exit;
  if (Bitstream[12] = '1') and
     (Length(Bitstream) <> 43) then Exit;

  RCString := '';
  Zero := '00';
  One  := '11';
  BiphaseZero := '01';
  BiphaseOne  := '10';

  RCString := '011111110010101001';
  for Index := 10 to 11 do
    if Bitstream[Index] = '0' then
      RCString := RCString + Zero
    else
      RCString := RCString + One;
  if Bitstream[12] = '0' then
    RCString := RCString + BiphaseZero
  else
    RCString := RCString + BiphaseOne;
  for Index := 13 to Length(Bitstream) do
    if Bitstream[Index] = '0' then
      RCString := RCString + BiphaseZero
    else
      RCString := RCString + BiphaseOne;

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <HalbitBitstream> Bitstream (represented as half bits)
            <Parameters>      Parameters
  Returns : <Result>          Result string
                              Empty if invalid

  Descript: Create Pronto ASCII code from bitstream
  Notes   : -
 ------------------------------------------------------------------------------}
function HalfbitBitstreamToProntoCodeRC6A(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
begin
  if Parameters[rptFrequency] = 0 then
    Parameters[rptFrequency] := CRCFrequency;
  Result := ZeroOneSequences(HalfbitBitstream, Parameters[rptDelay],
                             Parameters[rptFrequency], CRCBitTime, Parameters[rptRepeat]<>0);
end;


{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC6A code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC6A(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
begin
  RCString := ParametersToBitstreamRC6A(Parameters);
  RCString := BitstreamToHalfbitBitstreamRC6A(RCString);
  Result   := HalfbitBitstreamToProntoCodeRC6A(RCString, Parameters);
end;


end.






