{****************************************************************************}
{ Program: RC6.pas                                                           }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ RC6 functions/procedures                                                   }
{                                                                            }
{ Version                                                                    }
{  1.00   20020901  - First release                                          }
{****************************************************************************}
unit RC6;

interface

uses
  Windows;

{$I PRONTORCX.INC}

  function GenerateCodeRC6(Parameters: TRCParameters): AnsiString;

  function HalfbitBitstreamToProntoCodeRC6(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
  function ParametersToBitstreamRC6(Parameters: TRCParameters): AnsiString;
  function BitstreamToHalfbitBitstreamRC6(Bitstream: AnsiString): AnsiString;

implementation

uses
  RemoteGenerics;

const
  // Settings for generating RC5/RC5X/RC6/RC6A code
  CRCFrequency   = 36000;              // Default carrier frequency
  CRCBitTime     = 889 div 2;          // Bit time in us
  CRCDelay       = 69;                 // Default 69 ms delay after codes

// Format of a RC6 sequence
// Bitstream:
// hhhhhhhh TT SSSSSSSS CCCCCCCC
//
// hhhhhhhh  Header data (16)
//            '1111110010010101' (NOT to be biphase encoded!)
//  TT        Toggle (4)
//            '0011' or          (NOT to be biphase encoded!)
//            '1100'             (NOT to be biphase encoded!)
//  SSSSSSSS  System bits  (8)   (To be biphase encoded)
//  CCCCCCCC  Command bits (8)   (To be biphase encoded)
//
//  ONLY system and command bits are biphase encoded:
//  0 -> 01
//  1 -> 10               Note the inverted nature in comparison with RC5

{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single bitstream code from the parameters
  Notes   : -
 ------------------------------------------------------------------------------}
function ParametersToBitstreamRC6(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
  Zero    : char;
  One     : char;
begin
  Result    := '';
  if Parameters[rptAddress] > 255   then Exit;
  if Parameters[rptCommand] > 255  then Exit;

  RCString := '';
  Zero := '0';
  One  := '1';

  RCString := '00000000';
  if Parameters[rptToggle]<>0 then
    RCString := RCString + '10'
  else
    RCString := RCString + '01';

  RCString := RCString + EncodeBits(Parameters[rptAddress], 7, 0, Zero, One);
  RCString := RCString + EncodeBits(Parameters[rptCommand], 7, 0, Zero, One);

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <Bitstream>      Bitstream to convert
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Convert a bitstream into an (halfbit) encoded bitstream (bi-phase encoded)
  Notes   : -
 ------------------------------------------------------------------------------}
function BitstreamToHalfbitBitstreamRC6(Bitstream: AnsiString): AnsiString;
var
  RCString   : AnsiString;
  Zero       : string[2];
  One        : string[2];
  BiphaseZero: string[2];
  BiphaseOne : string[2];
  Index      : Integer;
begin
  Result    := '';
  // Sanity checks on data and length
  for Index := 1 to Length(Bitstream) do
    if not (BitStream[Index] in ['0', '1']) then
      Exit;
  if Length(Bitstream) <> 26 then
    Exit;

  RCString := '';
  Zero := '00';
  One  := '11';
  BiphaseZero := '01';
  BiphaseOne  := '10';

  RCString := '1111110010010101';
  for Index := 9 to 10 do
    if Bitstream[Index] = '0' then
      RCString := RCString + Zero
    else
      RCString := RCString + One;
  for Index := 11 to 26 do
    if Bitstream[Index] = '0' then
      RCString := RCString + BiphaseZero
    else
      RCString := RCString + BiphaseOne;

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <HalbitBitstream> Bitstream (represented as half bits)
            <Parameters>      Parameters
  Returns : <Result>          Result string
                              Empty if invalid

  Descript: Create Pronto ASCII code from bitstream
  Notes   : -
 ------------------------------------------------------------------------------}
function HalfbitBitstreamToProntoCodeRC6(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
begin
  if Parameters[rptFrequency] = 0 then
    Parameters[rptFrequency] := CRCFrequency;
  Result := ZeroOneSequences(HalfbitBitstream, Parameters[rptDelay],
                             Parameters[rptFrequency], CRCBitTime, Parameters[rptRepeat]<>0);
end;


{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC6 code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC6(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
begin
  RCString := ParametersToBitstreamRC6(Parameters);
  RCString := BitstreamToHalfbitBitstreamRC6(RCString);
  Result   := HalfbitBitstreamToProntoCodeRC6(RCString, Parameters);
end;


end.

