{****************************************************************************}
{ Program: RC5x.pas                                                          }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ RC5x functions/procedures                                                  }
{                                                                            }
{ Version                                                                    }
{  1.00   20020901  - First release                                          }
{****************************************************************************}
unit RC5x;

interface

uses
  Windows;

{$I PRONTORCX.INC}

  function GenerateCodeRC5x(Parameters: TRCParameters): AnsiString;

  function HalfbitBitstreamToProntoCodeRC5x(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
  function ParametersToBitstreamRC5x(Parameters: TRCParameters): AnsiString;
  function BitstreamToHalfbitBitstreamRC5x(Bitstream: AnsiString): AnsiString;

implementation

uses
  RemoteGenerics;

const
  // Settings for generating RC5/RC5X/RC6/RC6A code
  CRCFrequency   = 36000;              // Default carrier frequency
  CRCBitTime     = 889;                // (BitTime in us
  CRCDelay       = 69;                 // Default 69 ms delay after codes

// Format of a RC5X sequence
// Bitstream:
//  ss T SSSSS dd CCCCCC DDDDDD
//
//  ss = 10   Add 64 to command      (To be biphase encoded)
//  ss = 11   Use command as it is   (To be biphase encoded)
//  T         Toggle bit             (To be biphase encoded)
//  SSSSS     System bits (5)        (To be biphase encoded)
//  dddd      Divider bits (2)       (NOT to be biphase encoded)
//            '00'
//  CCCCCC    Command bits (6)       (To be biphase encoded)
//  DDDDDD    Data bits (6)          (To be biphase encoded)
//
//  ALL bits are biphase encoded, except for the dddd bits:
//  0 -> 10
//  1 -> 01
//
//  dddd bits encoded:
//  0 -> 00


{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single bitstream code from the parameters
  Notes   : -
 ------------------------------------------------------------------------------}
function ParametersToBitstreamRC5x(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
  Zero    : char;
  One     : char;
begin
  Result    := '';
  if Parameters[rptAddress] > 31   then Exit;
  if Parameters[rptCommand] > 127  then Exit;
  if Parameters[rptData]    > 63   then Exit;

  RCString := '';
  Zero := '0';
  One  := '1';

  if Parameters[rptCommand] > 63 then
    RCString := RCString + EncodeBits(2, 1, 0, Zero, One)
  else
    RCString := RCString + EncodeBits(3, 1, 0, Zero, One);
  if Parameters[rptToggle]<>0 then
    RCString := RCString + EncodeBits(1, 0, 0, Zero, One)
  else
    RCString := RCString + EncodeBits(0, 0, 0, Zero, One);
  RCString := RCString + EncodeBits(Parameters[rptAddress], 4, 0, Zero, One);
  RCString := RCString + '00';
  RCString := RCString + EncodeBits(Parameters[rptCommand], 5, 0, Zero, One);
  RCString := RCString + EncodeBits(Parameters[rptData]   , 5, 0, Zero, One);

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <Bitstream>      Bitstream to convert
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Convert a bitstream into an (halfbit) encoded bitstream (bi-phase encoded)
  Notes   : -
 ------------------------------------------------------------------------------}
function BitstreamToHalfbitBitstreamRC5x(Bitstream: AnsiString): AnsiString;
var
  RCString   : AnsiString;
  Zero       : string[2];
  One        : string[2];
  BiphaseZero: string[2];
  BiphaseOne : string[2];
  Index      : Integer;
begin
  Result    := '';
  // Sanity checks on data and length
  for Index := 1 to Length(Bitstream) do
    if not (BitStream[Index] in ['0', '1']) then
      Exit;
  if Length(Bitstream) <> 22 then
    Exit;

  RCString := '';
  Zero := '00';
  One  := '11';
  BiphaseZero := '10';
  BiphaseOne  := '01';

  for Index := 1 to 8 do
    if Bitstream[Index] = '0' then
      RCString := RCString + BiphaseZero
    else
      RCString := RCString + BiphaseOne;
  for Index := 9 to 10 do
    if Bitstream[Index] = '0' then
      RCString := RCString + Zero
    else
      RCString := RCString + One;
  for Index := 11 to 22 do
    if Bitstream[Index] = '0' then
      RCString := RCString + BiphaseZero
    else
      RCString := RCString + BiphaseOne;

  Result := RCString;
end;


{------------------------------------------------------------------------------
  Params  : <HalbitBitstream> Bitstream (represented as half bits)
            <Parameters>      Parameters
  Returns : <Result>          Result string
                              Empty if invalid

  Descript: Create Pronto ASCII code from bitstream
  Notes   : -
 ------------------------------------------------------------------------------}
function HalfbitBitstreamToProntoCodeRC5x(HalfbitBitstream: AnsiString; Parameters: TRCParameters): AnsiString;
begin
  if Parameters[rptFrequency] = 0 then
    Parameters[rptFrequency] := CRCFrequency;
  Result := ZeroOneSequences(HalfbitBitstream, Parameters[rptDelay],
                             Parameters[rptFrequency], CRCBitTime, Parameters[rptRepeat]<>0);
end;


{------------------------------------------------------------------------------
  Params  : <Parameters>     Parameters
  Returns : <Result>         Result string
                             Empty if invalid

  Descript: Create a single RC5x code for Pronto.
  Notes   : -
 ------------------------------------------------------------------------------}
function GenerateCodeRC5x(Parameters: TRCParameters): AnsiString;
var
  RCString: AnsiString;
begin
  RCString := ParametersToBitstreamRC5x(Parameters);
  RCString := BitstreamToHalfbitBitstreamRC5x(RCString);
  Result   := HalfbitBitstreamToProntoCodeRC5x(RCString, Parameters);
end;

end.
