{****************************************************************************}
{ Program: ProntoRc5x.pas                                                    }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ Component which creates RC5x command for the Pronto                        }
{                                                                            }
{ Version                                                                    }
{  1.00   20020818  - First release                                          }
{****************************************************************************}
unit ProntoRC5x;

interface

uses
  Classes, RemoteGenerics, RC5x, SysUtils, Windows;

type
  TProntoRC5x = class(TComponent)
  private
    FFrequency        : Dword;
    FDelay            : Dword;
    FToggle           : Boolean;
    FRepeat           : Boolean;
    FSystem           : TRCSystemTypeRCx;
    FCommand          : Dword;
    FData             : Dword;
    FShortformRC      : AnsiString;
    FShortform        : AnsiString;
    FLearned          : AnsiString;
    FBitstream        : AnsiString;
    FEncodedBitstream : AnsiString;
    FOnChangeCallback : TNotifyEvent;            // Event to be notified when generated code has changed
    procedure SetFrequency(Value: Dword);
    function  GetFrequency: Dword;
    procedure SetDelay(Value: Dword);
    function  GetDelay: Dword;
    procedure SetToggle(Value: Boolean);
    function  GetToggle: Boolean;
    procedure SetRepeat(Value: Boolean);
    function  GetRepeat: Boolean;
    procedure SetSystem(Value: TRCSystemTypeRCx);
    function  GetSystem: TRCSystemTypeRCx;
    procedure SetSystemNr(Value: Dword);
    function  GetSystemNr: Dword;
    procedure SetCommand(Value: Dword);
    function  GetCommand: Dword;
    procedure SetData(Value: Dword);
    function  GetData: Dword;
    procedure SetShortformRC(Value: AnsiString);
    function  GetShortformRC: AnsiString;
    procedure SetShortform(Value: AnsiString);
    function  GetShortform: AnsiString;
    procedure SetLearned(Value: AnsiString);
    function  GetLearned: AnsiString;
    procedure SetBitstream(Value: AnsiString);
    function  GetBitstream: AnsiString;
    procedure SetEncodedBitstream(Value: AnsiString);
    function  GetEncodedBitstream: AnsiString;
    procedure SetOnChange(Value: TNotifyEvent);
    function  GetOnChange: TNotifyEvent;
    procedure UpdateGeneratedCode;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  published
    { Published declarations }
    property Frequency       : Dword            read GetFrequency        write SetFrequency default 36000;
    property Delay           : Dword            read GetDelay            write SetDelay     default 69;
    property Toggle          : Boolean          read GetToggle           write SetToggle    default False;
    property RepeatCode      : Boolean          read GetRepeat           write SetRepeat    default False;
    property System          : TRCSystemTypeRCx read GetSystem           write SetSystem    default rstTelevision1;
    property SystemNr        : Dword            read GetSystemNr         write SetSystemNr  default ord(rstTelevision1);
    property Command         : Dword            read GetCommand          write SetCommand   default 0;
    property Data            : Dword            read GetData             write SetData      default 0;
    property ShortformRC     : AnsiString       read GetShortformRC      write SetShortformRC;
    property Shortform       : AnsiString       read GetShortform        write SetShortform;
    property Learned         : AnsiString       read GetLearned          write SetLearned;
    property Bitstream       : AnsiString       read GetBitstream        write SetBitstream;
    property EncodedBitstream: AnsiString       read GetEncodedBitstream write SetEncodedBitstream;
    property OnChange        : TNotifyEvent     read GetOnChange         write SetOnChange  default nil;
  end;

  procedure Register;

implementation


{------------------------------------------------------------------------------
  Params   : <AOwner> Owner of component
  Return   : -

  Descript : Create the component
  Notes    : -
 ------------------------------------------------------------------------------}
constructor TProntoRC5x.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Frequency   := 36000;
  Delay       := 69;
  Toggle      := False;
  RepeatCode  := False;
  System      := rstTelevision1;
  Command     := 0;
  Data        := 0;
  FShortformRC:= '';
  FShortform  := '';
  FLearned    := '';
  OnChange    := nil;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Destroy the component
  Notes    : -
 ------------------------------------------------------------------------------}
destructor TProntoRC5x.Destroy;
begin
  {}
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Update shortform and learned code fields
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.UpdateGeneratedCode;
var
  OldCode    : AnsiString;
  CarrierUnit: Word;
  Parameters : TRCParameters;
begin
  if Frequency < 64 then
    Frequency := 64;
  CarrierUnit := round(1E6/(Frequency  * CPFUnit));       // Actual timing unit

  Parameters[rptAddress]   := SystemNr;
  Parameters[rptCommand]   := Command;
  Parameters[rptData]      := Data;
  Parameters[rptDelay]     := Delay;
  if RepeatCode then
    Parameters[rptRepeat]  := 1
  else
    Parameters[rptRepeat]  := 0;
  Parameters[rptFrequency] := Frequency;
  if Toggle then
    Parameters[rptToggle]  := 1
  else
    Parameters[rptToggle]  := 0;

  FShortFormRC      := format('%s %d %d %d', ['RC5x', SystemNr, Command, Data]);
  FShortForm        := format('%4x %.4x %.4x %.4x %.4x %.4x %.4x %.4x', [$5001, CarrierUnit, 0, 2, SystemNr, Command, Data, 0]);
  FBitstream        := ParametersToBitstreamRC5x(RC5x.TRCParameters(Parameters));
  FEncodedBitstream := BitstreamToHalfbitBitstreamRC5x(FBitstream);
  OldCode := FLearned;
  FLearned          := GenerateCodeRC5x(RC5x.TRCParameters(Parameters));
  if (FLearned <> OldCode) then
    if assigned(FOnChangeCallback) then
      FOnChangeCallback(nil);
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Frequency in Hz
                      0 for default frequency
  Return   : -

  Descript : Set carrier frequency
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetFrequency(Value: Dword);
begin
  if Value < 64 then                                       // 1E6/(65535*0.241246) = minimum
    Exit;
  FFrequency := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Frequency in Hz

  Descript : Get carrier frequency
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetFrequency: Dword;
begin
  Result := FFrequency;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Delay in ms
  Return   : -

  Descript : Set delay
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetDelay(Value: Dword);
begin
  if Value > 5000 then
    Exit;
  FDelay := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Delay in ms

  Descript : Get delay
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetDelay: Dword;
begin
  Result := FDelay;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Toggle
  Return   : -

  Descript : Set toggle
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetToggle(Value: Boolean);
begin
  FToggle := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Toggle

  Descript : Get toggle
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetToggle: Boolean;
begin
  Result := FToggle;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Repeat
  Return   : -

  Descript : Set repeat
  Notes    : Repeat is only avaialable in the learned code format
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetRepeat(Value: Boolean);
begin
  FRepeat := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Repeat

  Descript : Get repeat
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetRepeat: Boolean;
begin
  Result := FRepeat;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  System type
  Return   : -

  Descript : Set system type
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetSystem(Value: TRCSystemTypeRCx);
begin
  FSystem := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  System type

  Descript : Get system type
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetSystem: TRCSystemTypeRCx;
begin
  Result := FSystem;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  System type
  Return   : -

  Descript : Set system type
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetSystemNr(Value: Dword);
begin
  if TRCSystemTypeRCx(Value) in [low(FSystem)..high(FSystem)] then
    FSystem := TRCSystemTypeRCx(Value);
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  System type

  Descript : Get system type
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetSystemNr: Dword;
begin
  Result := Ord(FSystem);
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Command
  Return   : -

  Descript : Set command
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetCommand(Value: Dword);
begin
  if Value > 127 then
    Exit;
  FCommand := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Command

  Descript : Get command
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetCommand: Dword;
begin
  Result := FCommand;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Data
  Return   : -

  Descript : Set data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetData(Value: Dword);
begin
  if Value > 63 then
    Exit;
  FData := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Data
  Descript : Get data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetData: Dword;
begin
  Result := FData;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Shortform
  Return   : -

  Descript : Set shortform code data (i.e. "RC5 0 0")
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetShortformRC(Value: AnsiString);
begin
  FShortFormRC := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Shortform code (ASCII)

  Descript : Get shortform code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetShortformRC: AnsiString;
begin
  Result := FShortformRC;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Shortform
  Return   : -

  Descript : Set shortform code data (i.e. "5000 xxxx")
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetShortform(Value: AnsiString);
begin
  FShortForm := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Shortform code (ASCII)

  Descript : Get shortform code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetShortform: AnsiString;
begin
  Result := FShortform;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Learned code (ASCII)
  Return   : -

  Descript : Set learned code data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetLearned(Value: AnsiString);
begin
  FLearned := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Learned code (ASCII)

  Descript : Get learned code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetLearned: AnsiString;
begin
  Result := FLearned;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Bitstream code (ASCII)
  Return   : -

  Descript : Set bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetBitstream(Value: AnsiString);
begin
  FBitstream := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Bitstream code (ASCII)

  Descript : Get bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetBitstream: AnsiString;
begin
  Result := FBitstream;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Encoded bitstream code (ASCII)
  Return   : -

  Descript : Set encoded bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetEncodedBitstream(Value: AnsiString);
begin
  FEncodedBitstream := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Encoded bitstream code (ASCII)

  Descript : Get encoded bitstream data
  Notes    : This is the bitstream with the relevant parts encoded
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetEncodedBitstream: AnsiString;
begin
  Result := FEncodedBitstream;
end;


{------------------------------------------------------------------------------
  Params   : <Value> Callback
  Return   : -

  Descript : Set the OnChange event
  Notes    :
 ------------------------------------------------------------------------------}
procedure TProntoRC5x.SetOnChange(Value: TNotifyEvent);
begin
  FOnChangeCallback := Value;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result> Callback

  Descript : Get the OnChange event
  Notes    :
 ------------------------------------------------------------------------------}
function TProntoRC5x.GetOnChange: TNotifyEvent;
begin
  Result := FOnChangeCallback;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Register component
  Notes    : -
 ------------------------------------------------------------------------------}
procedure Register;
begin
  RegisterComponents('Pronto', [TProntoRC5x]);
end;

end.
