{****************************************************************************}
{ Program: ProntoRc.pas                                                      }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ Component which creates remote control commands for the Pronto             }
{                                                                            }
{ Version                                                                    }
{  1.00   20020825  - First release                                          }
{****************************************************************************}
unit ProntoRC;

interface

uses
  Classes, RC5, RC5x, RC6, RC6a, SysUtils, Windows;

{$I PRONTORCX.INC}

type
  TRCType = (rctUserDefined, rctRC5, rctRC5x, rctRC6, rctRC6A);

  TProntoRC = class(TComponent)
  private
    FRCType           : TRCType;
    FFrequency        : Dword;
    FDelay            : Dword;
    FToggle           : Boolean;
    FRepeat           : Boolean;
    FSystem           : Dword;
    FCommand          : Dword;
    FCustomer         : Dword;
    FData             : Dword;
    FShortformRC      : AnsiString;
    FShortform        : AnsiString;
    FLearned          : AnsiString;
    FBitstream        : AnsiString;
    FEncodedBitstream : AnsiString;
    FOnChangeCallback : TNotifyEvent;            // Event to be notified when generated code has changed
    procedure SetRCType(Value: TRCType);
    function  GetRCType: TRCType;
    procedure SetFrequency(Value: Dword);
    function  GetFrequency: Dword;
    procedure SetDelay(Value: Dword);
    function  GetDelay: Dword;
    procedure SetToggle(Value: Boolean);
    function  GetToggle: Boolean;
    procedure SetRepeat(Value: Boolean);
    function  GetRepeat: Boolean;
    procedure SetSystem(Value: TRCSystemTypeRCx);
    function  GetSystem: TRCSystemTypeRCx;
    procedure SetSystemNr(Value: Dword);
    function  GetSystemNr: Dword;
    procedure SetCommand(Value: Dword);
    function  GetCommand: Dword;
    procedure SetCustomer(Value: Dword);
    function  GetCustomer: Dword;
    procedure SetData(Value: Dword);
    function  GetData: Dword;
    procedure SetShortformRC(Value: AnsiString);
    function  GetShortformRC: AnsiString;
    procedure SetShortform(Value: AnsiString);
    function  GetShortform: AnsiString;
    procedure SetLearned(Value: AnsiString);
    function  GetLearned: AnsiString;
    procedure SetBitstream(Value: AnsiString);
    function  GetBitstream: AnsiString;
    procedure SetEncodedBitstream(Value: AnsiString);
    function  GetEncodedBitstream: AnsiString;
    procedure SetOnChange(Value: TNotifyEvent);
    function  GetOnChange: TNotifyEvent;
    procedure UpdateGeneratedCode;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  published
    { Published declarations }
    property RCType          : TRCType          read GetRCType            write SetRCType    default rctUserDefined;
    property Frequency       : Dword            read GetFrequency         write SetFrequency default 36000;
    property Delay           : Dword            read GetDelay             write SetDelay     default 69;
    property Toggle          : Boolean          read GetToggle            write SetToggle    default False;
    property RepeatCode      : Boolean          read GetRepeat            write SetRepeat    default False;
    property System          : TRCSystemTypeRCx read GetSystem            write SetSystem    default rstTelevision1;
    property SystemNr        : Dword            read GetSystemNr          write SetSystemNr  default ord(rstTelevision1);
    property Command         : Dword            read GetCommand           write SetCommand   default 0;
    property Data            : Dword            read GetData              write SetData      default 0;
    property Customer        : Dword            read GetCustomer          write SetCustomer  default 0;
    property ShortformRC     : AnsiString       read GetShortformRC       write SetShortformRC;
    property Shortform       : AnsiString       read GetShortform         write SetShortform;
    property Learned         : AnsiString       read GetLearned           write SetLearned;
    property Bitstream       : AnsiString       read GetBitstream         write SetBitstream;
    property EncodedBitstream: AnsiString       read GetEncodedBitstream  write SetEncodedBitstream;
    property OnChange        : TNotifyEvent     read GetOnChange          write SetOnChange  default nil;
  end;

  procedure Register;

implementation


{------------------------------------------------------------------------------
  Params   : <AOwner> Owner of component
  Return   : -

  Descript : Create the component
  Notes    : -
 ------------------------------------------------------------------------------}
constructor TProntoRC.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  RCType      := rctUserdefined;
  Frequency   := 36000;
  Delay       := 69;
  Toggle      := False;
  RepeatCode  := False;
  System      := rstTelevision1;
  Command     := 0;
  Customer    := 0;
  Data        := 0;
  FShortformRC:= '';
  FShortform  := '';
  FLearned    := '';
  OnChange    := nil;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Destroy the component
  Notes    : -
 ------------------------------------------------------------------------------}
destructor TProntoRC.Destroy;
begin
  {}
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Update shortform and learned code fields
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.UpdateGeneratedCode;
var
  OldCode: AnsiString;
  CarrierUnit: Dword;
  Parameters : TRCParameters;
begin
  if Frequency < 64 then
    Frequency := 64;
  CarrierUnit := round(1E6/(Frequency  * 0.241246));       // Actual timing unit
  OldCode := FLearned;

  Parameters[rptAddress]   := SystemNr;
  Parameters[rptCommand]   := Command;
  Parameters[rptCustomer]  := Customer;
  Parameters[rptData]      := Data;
  Parameters[rptDelay]     := Delay;
  if RepeatCode then
    Parameters[rptRepeat]  := 1
  else
    Parameters[rptRepeat]  := 0;
  Parameters[rptFrequency] := Frequency;
  if Toggle then
    Parameters[rptToggle]  := 1
  else
    Parameters[rptToggle]  := 0;

  // The user defined type leaves all generated fields alone
  case RCType of
    rctUserDefined:
      begin
        OldCode := FLearned + 'X';               // Just to activate event
      end;
    rctRC5:
      begin
        FShortFormRC      := format('%s %d %d', ['RC5', SystemNr, Command]);
        FShortForm        := format('%4x %.4x %.4x %.4x %.4x %.4x', [$5000, CarrierUnit, 0, 1, SystemNr, Command]);
        FBitstream        := ParametersToBitstreamRC5(RC5.TRCParameters(Parameters));
        FEncodedBitstream := BitstreamToHalfbitBitstreamRC5(FBitstream);
        FLearned          := GenerateCodeRC5(RC5.TRCParameters(Parameters));
      end;
    rctRC5x:
      begin
        FShortFormRC      := format('%s %d %d %d', ['RC5x', SystemNr, Command, Data]);
        FShortForm        := format('%4x %.4x %.4x %.4x %.4x %.4x %.4x %.4x', [$5001, CarrierUnit, 0, 2, SystemNr, Command, Data, 0]);
        FBitstream        := ParametersToBitstreamRC5x(RC5x.TRCParameters(Parameters));
        FEncodedBitstream := BitstreamToHalfbitBitstreamRC5x(FBitstream);
        FLearned          := GenerateCodeRC5x(RC5x.TRCParameters(Parameters));
      end;
    rctRC6:
      begin
        FShortFormRC      := format('%s %d %d', ['RC6', SystemNr, Command]);
        FShortForm        := format('%4x %.4x %.4x %.4x %.4x %.4x', [$6000, CarrierUnit, 0, 1, SystemNr, Command]);
        FBitstream        := ParametersToBitstreamRC6(RC6.TRCParameters(Parameters));
        FEncodedBitstream := BitstreamToHalfbitBitstreamRC6(FBitstream);
        FLearned          := GenerateCodeRC6(RC6.TRCParameters(Parameters));
      end;
    rctRC6A:
      begin
        FShortFormRC      := format('%s %d %d', ['RC6A', SystemNr, Command]);
        FShortForm        := format('%4x %.4x %.4x %.4x %.4x %.4x %.4x %.4x', [$6001, CarrierUnit, 0, 2, Customer, SystemNr, Command, 0]);
        FBitstream        := ParametersToBitstreamRC6A(RC6A.TRCParameters(Parameters));
        FEncodedBitstream := BitstreamToHalfbitBitstreamRC6A(FBitstream);
        FLearned          := GenerateCodeRC6A(RC6A.TRCParameters(Parameters));
      end;
  end;
  if (FLearned <> OldCode) then
    if assigned(FOnChangeCallback) then
      FOnChangeCallback(nil);
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Type of remote control
  Return   : -

  Descript : Set type of remote control code
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetRCType(Value: TRCType);
begin
  FRCType := Value;
  // Since the system and command range are specific to a RC type, 'refresh' them
  SetSystemNr(FSystem);
  SetCommand(FCommand);
  case FRCType of
    rctUserDefined:
      begin
        FShortFormRC      := '';
        FShortForm        := '';
        FBitstream        := '';
        FEncodedBitstream := '';
        FLearned          := '';
      end;
  end;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Type of remote control

  Descript : Get type of remote control code
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetRCType: TRCType;
begin
  Result := FRCType;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Frequency in Hz
                      0 for default frequency
  Return   : -

  Descript : Set carrier frequency
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetFrequency(Value: Dword);
begin
  if Value < 64 then                                       // 1E6/(65535*0.241246) = minimum
    Value := 64;
  FFrequency := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Frequency in Hz

  Descript : Get carrier frequency
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetFrequency: Dword;
begin
  Result := FFrequency;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Delay in ms
  Return   : -

  Descript : Set delay
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetDelay(Value: Dword);
begin
  if Value > 5000 then
    Value := 5000;
  FDelay := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Delay in ms

  Descript : Get delay
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetDelay: Dword;
begin
  Result := FDelay;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Toggle
  Return   : -

  Descript : Set toggle
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetToggle(Value: Boolean);
begin
  FToggle := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Toggle

  Descript : Get toggle
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetToggle: Boolean;
begin
  Result := FToggle;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Repeat
  Return   : -

  Descript : Set repeat
  Notes    : Repeat is only available in the learned code format
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetRepeat(Value: Boolean);
begin
  FRepeat := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Repeat

  Descript : Get repeat
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetRepeat: Boolean;
begin
  Result := FRepeat;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  System type
  Return   : -

  Descript : Set system type
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetSystem(Value: TRCSystemTypeRCx);
begin
  FSystem := Ord(Value);
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  System type

  Descript : Get system type
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetSystem: TRCSystemTypeRCx;
begin
  if TRCSystemTypeRCx(FSystem) > high(TRCSystemTypeRCx) then
    Result := rstTelevision1
  else
    Result := TRCSystemTypeRCx(FSystem);
end;


{------------------------------------------------------------------------------
  Params   : <Value>  System type
  Return   : -

  Descript : Set system type
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetSystemNr(Value: Dword);
begin
  if Value > 255 then
    Value := 255;
  if (RCType = rctRC5) and
     (Value > 31) then
    Value := 31;
  if (RCType = rctRC5x) and
     (Value > 31) then
    Value := 31;
  FSystem := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  System type

  Descript : Get system type
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetSystemNr: DWord;
begin
  Result := Ord(FSystem);
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Command
  Return   : -

  Descript : Set command
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetCommand(Value: Dword);
begin
  if Value > 255 then
    Value := 255;
  if (RCType = rctRC5) and
     (Value > 127) then
    Value := 127;
  if (RCType = rctRC5x) and
     (Value > 127) then
    Value := 127;
  FCommand := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Command

  Descript : Get command
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetCommand: Dword;
begin
  Result := FCommand;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Customer
  Return   : -

  Descript : Set customer code
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetCustomer(Value: Dword);
begin
  if (Value > 127) and (Value < 32768) then
    Value := 127;
  if Value > 65535 then
    Value := 65535;
  FCustomer := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Customer

  Descript : Get customer code
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetCustomer: Dword;
begin
  Result := FCustomer;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Data
  Return   : -

  Descript : Set data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetData(Value: Dword);
begin
  if Value > 63 then
    Value := 63;
  FData := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Data
  Descript : Get data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetData: Dword;
begin
  Result := FData;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Shortform
  Return   : -

  Descript : Set shortform code data (i.e. "RC5 0 0")
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetShortformRC(Value: AnsiString);
begin
  FShortFormRC := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Shortform code (ASCII)

  Descript : Get shortform code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetShortformRC: AnsiString;
begin
  Result := FShortformRC;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Shortform
  Return   : -

  Descript : Set shortform code data (i.e. "5000 xx")
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetShortform(Value: AnsiString);
begin
  FShortForm := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Shortform code (ASCII)

  Descript : Get shortform code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetShortform: AnsiString;
begin
  Result := FShortform;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Learned code (ASCII)
  Return   : -

  Descript : Set learned code data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetLearned(Value: AnsiString);
begin
  FLearned := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Learned code (ASCII)

  Descript : Get learned code data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetLearned: AnsiString;
begin
  Result := FLearned;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Bitstream code (ASCII)
  Return   : -

  Descript : Set bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetBitstream(Value: AnsiString);
begin
  FBitstream := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Bitstream code (ASCII)

  Descript : Get bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
function TProntoRC.GetBitstream: AnsiString;
begin
  Result := FBitstream;
end;


{------------------------------------------------------------------------------
  Params   : <Value>  Encoded bitstream code (ASCII)
  Return   : -

  Descript : Set encoded bitstream data
  Notes    : -
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetEncodedBitstream(Value: AnsiString);
begin
  FEncodedBitstream := Value;
  UpdateGeneratedCode;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>  Encoded bitstream code (ASCII)

  Descript : Get encoded bitstream data
  Notes    : This is the bitstream with the relevant parts encoded
 ------------------------------------------------------------------------------}
function TProntoRC.GetEncodedBitstream: AnsiString;
begin
  Result := FEncodedBitstream;
end;


{------------------------------------------------------------------------------
  Params   : <Value> Callback
  Return   : -

  Descript : Set the OnChange event
  Notes    :
 ------------------------------------------------------------------------------}
procedure TProntoRC.SetOnChange(Value: TNotifyEvent);
begin
  FOnChangeCallback := Value;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result> Callback

  Descript : Get the OnChange event
  Notes    :
 ------------------------------------------------------------------------------}
function TProntoRC.GetOnChange: TNotifyEvent;
begin
  Result := FOnChangeCallback;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Register component
  Notes    : -
 ------------------------------------------------------------------------------}
procedure Register;
begin
  RegisterComponents('Pronto', [TProntoRC]);
end;

end.
