{****************************************************************************}
{ Program: GenerateRCx.pas                                                   }
{ Author : M.Majoor                                                          }
{----------------------------------------------------------------------------}
{ Generate a RC5/RC5x/RC6/RC6A code.                                         }
{ Note: Code is as straightforward as possible.....                          }
{                                                                            }
{ Command line parameters:                                                   }
{   1    2         3       4       5      6        7       8       9         }
{   RC5  Frequency Delay   Toggle  Repeat System   Command Format            }
{   RC5  System    Command Format                                            }
{   RC5X Frequency Delay   Toggle  Repeat System   Command Data    Format    }
{   RC5X System    Command Data    Format                                    }
{   RC6  Frequency Delay   Toggle  Repeat System   Command Format            }
{   RC6  System    Command Format                                            }
{   RC6A Frequency Delay   Toggle  Repeat Customer System  Command Format    }
{   RC6A Customer  System  Command Format                                    }
{   5000 0000      0000    0001    ssss   cccc     Format                    }
{   5000 ffff      0000    0001    ssss   cccc     Format                    }
{   5001 0000      0000    0002    ssss   cccc     dddd    0000    Format    }
{   5001 ffff      0000    0002    ssss   cccc     dddd    0000    Format    }
{   6000 0000      0000    0001    ssss   cccc     Format                    }
{   6000 ffff      0000    0001    ssss   cccc     Format                    }
{   6001 0000      0000    0002    uuuu   ssss     cccc    0000    Format    }
{   6001 ffff      0000    0002    uuuu   ssss     cccc    0000    Format    }
{                                                                            }
{ were:                                                                      }
{   Frequency == Frequency in Hz                                             }
{   Delay     == Delay in ms                                                 }
{   Toggle    == Toggle data is set  (1) or not (0)                          }
{   Repeat    == Repeat      is used (1) or not (0)                          }
{   Customer  == Customer, decimal number                                    }
{   System    == System, decimal number                                      }
{   Command   == Command, decimal number                                     }
{   Data      == Data, decimal number                                        }
{   Format    == Output format                                               }
{                0:  Learned format (eg. 0000 0070 .......)                  }
{                1:  Short form 1   (eg. RC5 0 0)                            }
{                2:  Short form 2   (eg. 5000 0000 0000 0001 0000 0000       }
{                3:  Bitstream                                               }
{                4:  Encoded bitstream                                       }
{                Add 10 to the output format to include the orginal data in  }
{                the output.                                                 }
{   ffff      == Carrier frequency in Pronto units, in hexadecimal           }
{   uuuu      == Customer, in hexadecimal                                    }
{   ssss      == System, in hexadecimal                                      }
{   cccc      == Command, in hexadecimal                                     }
{   dddd      == Data, in hexadecimal                                        }
{                                                                            }
{ Returned result code:                                                      }
{   0 = Success                                                              }
{   1 = Number of parameters not correct                                     }
{  1x = Error in parameter x                                                 }
{ If no error is detected then the result is being output on the console.    }
{ If the format type is >=10 then the original data is being output to the   }
{ console also.                                                              }
{                                                                            }
{ Note: If the 'Format' is omitted then the learned code format is used      }
{                                                                            }
{ Version                                                                    }
{  1.00   20030323 - First release                                           }
{  1.01   20050904 - RC6A did not put 'command' in correct property so       }
{                    generated data was always incorrect                     }
{****************************************************************************}
program GenerateRCx;
{$APPTYPE CONSOLE}

uses
  Clipbrd,
  ProntoRC,                            // Generic wrapper for RC5/RC5x/RC6/RC6A
  SysUtils,
  Windows;

{$R *.RES}

var
  ProntoControl: TProntoRC;
  SourceType   : string;
  Frequency    : Longint;
  Delay        : Longint;
  Toggle       : Longint;
  UseRepeat    : Longint;
  CustomerId   : Longint;
  SystemId     : Longint;
  Command      : Longint;
  Data         : Longint;
  FormatId     : Longint;
  Loop         : Integer;
  OriginalData : AnsiString;
  ReturnData   : AnsiString;
  ClipboardCount     : Integer;
  ClipboardParameters: array[1..9] of string;
  TheClipboard       : TClipboard;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Check clipboard for parameters
  Notes    : -
 ------------------------------------------------------------------------------}
procedure CheckClipboard;
var
  InputString: string;
  Partstring : string;
  SpacePos   : Integer;
  Loop       : Integer;
begin
  ClipboardCount := 0;
  for Loop := Low(ClipboardParameters) to High(ClipboardParameters) do
    ClipboardParameters[Loop] := '';

  if TheClipboard.HasFormat(CF_TEXT) then
  begin
    InputString := TheClipboard.AsText;
    while Length(InputString) > 0 do
    begin
      InputString := Trim(InputString);
      if Length(InputString) > 0 then
      begin
        SpacePos := Pos(' ', InputString);
        if SpacePos = 0 then
          SpacePos := Length(InputString);
        PartString := Copy(InputString, 1, SpacePos);
        PartString := Trim(PartString);          // We most likley copied a space
        Delete(InputString, 1, SpacePos);
        Inc(ClipboardCount);
        ClipboardParameters[ClipboardCount] := PartString;
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params   : <Index>        Parameter to check
  Return   : <Result>       Parameter string

  Descript : Equivalent to <ParamStr> but also uses clipboard
  Notes    : -
 ------------------------------------------------------------------------------}
function ParamStr2(Index: Integer): string;
begin
  if ParamCount > 0 then
    Result := ParamStr(Index)
  else
  begin
    if Index in [Low(ClipboardParameters)..High(ClipboardParameters)] then
      Result := ClipboardParameters[Index]
    else
      Result := '';
  end;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : <Result>       Number of parameters

  Descript : Equivalent to <ParamCount> but also inspect clipboard
  Notes    : -
 ------------------------------------------------------------------------------}
function ParamCount2: Integer;
begin
  if ParamCount > 0 then
    Result := ParamCount
  else
  begin
    Result := ClipboardCount;
  end;
end;


{------------------------------------------------------------------------------
  Params   : <TypeId>       0 = digital
                            1 = hexadecimal
             <ParameterNr>  Numbre of parameter
             <Data>         Converted data
  Return   : <Result>       True if valid result
                            False if conversion failed

  Descript : Convert parameter
  Notes    : -
 ------------------------------------------------------------------------------}
function ConvertParameter(TypeId: Byte; ParameterNr: Byte; var Data: Longint): Boolean;
var
  Error: Integer;
begin
  if TypeId = 0 then
    Val(ParamStr2(ParameterNr), Data, Error);
  if TypeId = 1 then
    Val('$' + ParamStr2(ParameterNr), Data, Error);
  if Error <> 0 then
    Result := False
  else
    Result := True;
end;


{------------------------------------------------------------------------------
  Params   : -
  Return   : -

  Descript : Application start
  Notes    : -
 ------------------------------------------------------------------------------}
begin
  ProntoControl := TProntoRC.Create(nil);                  // Create the object
  // Set all parameters to there defaults
  ReturnData := '';
  SourceType := '';
  ClipboardCount := 0;
  // Only use clipboard if no parameters passed ...
  if ParamCount = 0 then
  begin
    TheClipboard := Clipboard;
    CheckClipboard;
  end;

  if ParamCount2 > 0 then
    SourceType := UpperCase(ParamStr2(1));

  // RC5
  // RC5  Frequency Delay   Toggle  Repeat System   Command Format
  // RC5  System    Command Format
  if SourceType = 'RC5' then
  begin
    // Check number of parameters
    // Note: The 'Format' parameter is optional
    if not ParamCount2 in [3, 4, 7, 8] then
      Halt(1);
    // Set defaults
    ProntoControl.RCType     := rctRC5;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if (ParamCount2 = 3) or
       (ParamCount2 = 4) then
    begin
      if not ConvertParameter(0, 2, SystemId) then
        Halt(12);
      if not ConvertParameter(0, 3, Command) then
        Halt(13);
      if ParamCount2 = 4 then
        if not ConvertParameter(0, 4, FormatId) then
          Halt(14);
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
    end
    else
    begin
      if not ConvertParameter(0, 2, Frequency) then
        Halt(12);
      if not ConvertParameter(0, 3, Delay) then
        Halt(13);
      if not ConvertParameter(0, 4, Toggle) then
        Halt(14);
      if not ConvertParameter(0, 5, UseRepeat) then
        Halt(15);
      if not ConvertParameter(0, 6, SystemId) then
        Halt(16);
      if not ConvertParameter(0, 7, Command) then
        Halt(17);
      if ParamCount2 = 8 then
        if not ConvertParameter(0, 8, FormatId) then
          Halt(18);
      ProntoControl.Frequency := Frequency;
      ProntoControl.Delay     := Delay;
      if Toggle <> 0 then
        ProntoControl.Toggle := True
      else
        ProntoControl.Toggle := False;
      if UseRepeat <> 0 then
        ProntoControl.RepeatCode := True
      else
        ProntoControl.RepeatCode := False;
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
    end;
    SourceType := '';
  end;

  // RC5x
  // RC5X Frequency Delay   Toggle  Repeat System   Command Data    Format
  // RC5X System    Command Data    Format
  if SourceType = 'RC5X' then
  begin
    if not ParamCount2 in [4, 5, 8, 9] then
      Halt(1);
    ProntoControl.RCType     := rctRC5x;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if (ParamCount2 = 4) or
       (ParamCount2 = 5) then
    begin
      if not ConvertParameter(0, 2, SystemId) then
        Halt(12);
      if not ConvertParameter(0, 3, Command) then
        Halt(13);
      if not ConvertParameter(0, 4, Data) then
        Halt(14);
      if ParamCount2 = 5 then
        if not ConvertParameter(0, 5, FormatId) then
          Halt(15);
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
      ProntoControl.Data     := Data;
    end
    else
    begin
      if not ConvertParameter(0, 2, Frequency) then
        Halt(12);
      if not ConvertParameter(0, 3, Delay) then
        Halt(13);
      if not ConvertParameter(0, 4, Toggle) then
        Halt(14);
      if not ConvertParameter(0, 5, UseRepeat) then
        Halt(15);
      if not ConvertParameter(0, 6, SystemId) then
        Halt(16);
      if not ConvertParameter(0, 7, Command) then
        Halt(17);
      if not ConvertParameter(0, 8, Data) then
        Halt(18);
      if ParamCount2 = 9 then
        if not ConvertParameter(0, 9, FormatId) then
          Halt(19);
      ProntoControl.Frequency := Frequency;
      ProntoControl.Delay     := Delay;
      if Toggle <> 0 then
        ProntoControl.Toggle := True
      else
        ProntoControl.Toggle := False;
      if UseRepeat <> 0 then
        ProntoControl.RepeatCode := True
      else
        ProntoControl.RepeatCode := False;
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
      ProntoControl.Data     := Data;
    end;
    SourceType := '';
  end;

  // RC6
  // RC6  Frequency Delay   Toggle  Repeat System   Command Format
  // RC6  System    Command Format
  if SourceType = 'RC6' then
  begin
    if not ParamCount2 in [3, 4, 7, 8] then
      Halt(1);
    ProntoControl.RCType     := rctRC6;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if (ParamCount2 = 3) or
       (ParamCount2 = 4) then
    begin
      if not ConvertParameter(0, 2, SystemId) then
        Halt(12);
      if not ConvertParameter(0, 3, Command) then
        Halt(13);
      if ParamCount2 = 4 then
        if not ConvertParameter(0, 4, FormatId) then
          Halt(14);
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
    end
    else
    begin
      if not ConvertParameter(0, 2, Frequency) then
        Halt(12);
      if not ConvertParameter(0, 3, Delay) then
        Halt(13);
      if not ConvertParameter(0, 4, Toggle) then
        Halt(14);
      if not ConvertParameter(0, 5, UseRepeat) then
        Halt(15);
      if not ConvertParameter(0, 6, SystemId) then
        Halt(16);
      if not ConvertParameter(0, 7, Command) then
        Halt(17);
      if ParamCount2 = 8 then
        if not ConvertParameter(0, 8, FormatId) then
          Halt(18);
      ProntoControl.Frequency := Frequency;
      ProntoControl.Delay     := Delay;
      if Toggle <> 0 then
        ProntoControl.Toggle := True
      else
        ProntoControl.Toggle := False;
      if UseRepeat <> 0 then
        ProntoControl.RepeatCode := True
      else
        ProntoControl.RepeatCode := False;
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
    end;
    SourceType := '';
  end;

  // RC6A
  // RC6A Frequency Delay   Toggle  Repeat Customer System  Command Format
  // RC6A Customer  System  Command Format
  if SourceType = 'RC6A' then
  begin
    if not ParamCount2 in [4, 5, 8, 9] then
      Halt(1);
    ProntoControl.RCType     := rctRC6A;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if (ParamCount2 = 4) or
       (ParamCount2 = 5) then
    begin
      if not ConvertParameter(0, 2, CustomerId) then
        Halt(12);
      if not ConvertParameter(0, 3, SystemId) then
        Halt(13);
      if not ConvertParameter(0, 4, Command) then
        Halt(14);
      if ParamCount2 = 5 then
        if not ConvertParameter(0, 5, FormatId) then
          Halt(15);
      ProntoControl.Customer := CustomerId;
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
      ProntoControl.Data     := Data;
    end
    else
    begin
      if not ConvertParameter(0, 2, Frequency) then
        Halt(12);
      if not ConvertParameter(0, 3, Delay) then
        Halt(13);
      if not ConvertParameter(0, 4, Toggle) then
        Halt(14);
      if not ConvertParameter(0, 5, UseRepeat) then
        Halt(15);
      if not ConvertParameter(0, 6, CustomerId) then
        Halt(16);
      if not ConvertParameter(0, 7, SystemId) then
        Halt(17);
      if not ConvertParameter(0, 8, Command) then
        Halt(18);
      if ParamCount2 = 9 then
        if not ConvertParameter(0, 9, FormatId) then
          Halt(19);
      ProntoControl.Frequency := Frequency;
      ProntoControl.Delay     := Delay;
      if Toggle <> 0 then
        ProntoControl.Toggle := True
      else
        ProntoControl.Toggle := False;
      if UseRepeat <> 0 then
        ProntoControl.RepeatCode := True
      else
        ProntoControl.RepeatCode := False;
      ProntoControl.SystemNr := SystemId;
      ProntoControl.Command  := Command;
      ProntoControl.Data     := Data;
    end;
    SourceType := '';
  end;

  // 5000
  // 5000 0000      0000    0001    ssss   cccc     Format
  // 5000 ffff      0000    0001    ssss   cccc     Format
  if SourceType = '5000' then
  begin
    if not ParamCount2 in [6, 7] then
      Halt(1);
    ProntoControl.RCType     := rctRC5;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if not ConvertParameter(1, 2, Frequency) then
      Halt(12);
    if Frequency <> 0 then
    begin
      Frequency := round(1E6/(Frequency  * 0.241246));       // Actual frequency (source was in Pronto units)
      ProntoControl.Frequency := Frequency;
    end;
    // Next parameter MUST be 0
    if not ConvertParameter(1, 3, Frequency) then
      Halt(13);
    if Frequency <> 0 then
      Halt(13);
    // Next parameter MUST be 1
    if not ConvertParameter(1, 4, Frequency) then
      Halt(14);
    if Frequency <> 1 then
      Halt(14);
    if not ConvertParameter(1, 5, SystemId) then
      Halt(15);
    ProntoControl.SystemNr := SystemId;
    if not ConvertParameter(1, 6, Command) then
      Halt(16);
    ProntoControl.Command := Command;
    if ParamCount2 = 7 then
      if not ConvertParameter(0, 7, FormatId) then
        Halt(17);
    SourceType := '';
  end;

  // 5001
  // 5001 0000      0000    0002    ssss   cccc     dddd    0000    Format
  // 5001 ffff      0000    0002    ssss   cccc     dddd    0000    Format
  if SourceType = '5001' then
  begin
    if not ParamCount2 in [8, 9] then
      Halt(1);
    ProntoControl.RCType     := rctRC5x;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if not ConvertParameter(1, 2, Frequency) then
      Halt(12);
    if Frequency <> 0 then
    begin
      Frequency := round(1E6/(Frequency  * 0.241246));       // Actual frequency (source was in Pronto units)
      ProntoControl.Frequency := Frequency;
    end;

    // Next parameter MUST be 0
    if not ConvertParameter(1, 3, Frequency) then
      Halt(13);
    if Frequency <> 0 then
      Halt(13);
    // Next parameter MUST be 2
    if not ConvertParameter(1, 4, Frequency) then
      Halt(14);
    if Frequency <> 2 then
      Halt(14);
    if not ConvertParameter(1, 5, SystemId) then
      Halt(15);
    ProntoControl.SystemNr := SystemId;
    if not ConvertParameter(1, 6, Command) then
      Halt(16);
    ProntoControl.Command := Command;
    if not ConvertParameter(1, 7, Data) then
      Halt(17);
    ProntoControl.Data := Data;
    // Next parameter MUST be 0
    if not ConvertParameter(1, 8, Frequency) then
      Halt(18);
    if Frequency <> 0 then
      Halt(18);
    if ParamCount2 = 9 then
      if not ConvertParameter(0, 9, FormatId) then
        Halt(19);
    SourceType := '';
  end;

  // 6000
  // 6000 0000      0000    0001    ssss   cccc     Format
  // 6000 ffff      0000    0001    ssss   cccc     Format
  if SourceType = '6000' then
  begin
    if not ParamCount2 in [6, 7] then
      Halt(1);
    ProntoControl.RCType     := rctRC6;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if not ConvertParameter(1, 2, Frequency) then
      Halt(12);
    if Frequency <> 0 then
    begin
      Frequency := round(1E6/(Frequency  * 0.241246));       // Actual frequency (source was in Pronto units)
      ProntoControl.Frequency := Frequency;
    end;
    // Next parameter MUST be 0
    if not ConvertParameter(1, 3, Frequency) then
      Halt(13);
    if Frequency <> 0 then
      Halt(13);
    // Next parameter MUST be 1
    if not ConvertParameter(1, 4, Frequency) then
      Halt(14);
    if Frequency <> 1 then
      Halt(14);
    if not ConvertParameter(1, 5, SystemId) then
      Halt(15);
    ProntoControl.SystemNr := SystemId;
    if not ConvertParameter(1, 6, Command) then
      Halt(16);
    ProntoControl.Command := Command;
    if ParamCount2 = 7 then
      if not ConvertParameter(0, 7, FormatId) then
        Halt(17);
    SourceType := '';
  end;

  // 6001
  // 6001 0000      0000    0002    uuuu   ssss     cccc    0000    Format
  // 6001 ffff      0000    0002    uuuu   ssss     cccc    0000    Format
  if SourceType = '6001' then
  begin
    if not ParamCount2 in [8, 9] then
      Halt(1);
    ProntoControl.RCType     := rctRC6A;
    ProntoControl.Delay      := 69;
    ProntoControl.Frequency  := 36000;
    ProntoControl.Toggle     := False;
    ProntoControl.RepeatCode := True;
    if not ConvertParameter(1, 2, Frequency) then
      Halt(12);
    if Frequency <> 0 then
    begin
      Frequency := round(1E6/(Frequency  * 0.241246));       // Actual frequency (source is in Pronto units)
      ProntoControl.Frequency := Frequency;
    end;
    // Next parameter MUST be 0
    if not ConvertParameter(1, 3, Frequency) then
      Halt(13);
    if Frequency <> 0 then
      Halt(13);
    // Next parameter MUST be 2
    if not ConvertParameter(1, 4, Frequency) then
      Halt(14);
    if Frequency <> 2 then
      Halt(14);
    if not ConvertParameter(1, 5, CustomerId) then
      Halt(15);
    ProntoControl.Customer := CustomerId;
    if not ConvertParameter(1, 6, SystemId) then
      Halt(16);
    ProntoControl.SystemNr := SystemId;
    if not ConvertParameter(1, 7, Command) then
      Halt(17);
    ProntoControl.Command := Command;
    // Next parameter MUST be 0
    if not ConvertParameter(1, 8, Frequency) then
      Halt(18);
    if Frequency <> 0 then
      Halt(18);
    if ParamCount2 = 9 then
      if not ConvertParameter(0, 9, FormatId) then
        Halt(19);
    SourceType := '';
  end;
  // If source type parameter 1 not handled: report error
  if SourceType <> '' then
    Halt(11);

  case FormatId of
    1, 11: ReturnData := ProntoControl.ShortformRC;
    2, 12: ReturnData := ProntoControl.Shortform;
    3, 13: ReturnData := ProntoControl.Bitstream;
    4, 14: ReturnData := ProntoControl.EncodedBitstream;
    else ReturnData := ProntoControl.Learned;
  end;

  OriginalData := '';
  for Loop := 1 to ParamCount2 do
    OriginalData := OriginalData + ParamStr2(Loop) + ' ';
  Delete(OriginalData, Length(OriginalData), 1);
  // Now the result: either to the console or clipboard
  if ParamCount > 0 then
  begin
    if FormatId >= 10 then
      WriteLn(OriginalData);
    WriteLn(ReturnData);
  end
  else
  begin
    if FormatId >= 10 then
      ReturnData := OriginalData + #$0D+ #$0A + ReturnData + #0
    else
      ReturnData   := ReturnData + #0;
    TheClipboard.SetTextBuf(@ReturnData[1]);
  end;
  ProntoControl.Free;                                      // Remove the object
end.

