{******************************************************************************}
{ FileName............: ShowingStreamDataUnit001                               }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Buffering / Showing streams.                                                }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ The default settings should provide you with a video/audio of the german ARD }
{ program being transmitted. It is assumed the dish is facing Astra 19.2E and  }
{ that the PIDs have not changed....                                           }
{ If no video/audio is shown (without an error being reported) then check the  }
{ also the graphics setup (see note below).                                    }
{                                                                              }
{ Note: Only one DirectShow per process can be run ....                        }
{ Note: Debug not always possible because of DirectShow stuff ...              }
{ Note: The supplied VIDEO.GRF has no connected output pins. This is not       }
{       absolute necessary because these are automatically connected to the    }
{       'correct' video/audio filters.                                         }
{       These video/audio filters are typically part of a DVD viewer (like     }
{       InterVideo / PowerDVD etc. You can also connect these pins as required }
{       so they are forced to a specific connection. You need the FilterGraph  }
{       from DirectX 9 for this.                                               }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050412 - Initial release                                           }
{******************************************************************************}
unit ShowingStreamDataUnit001;

interface
uses
  Forms, ExtCtrls, StdCtrls, Mask, Controls, Classes, Gauges;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    mskSymbolRate: TMaskEdit;
    StaticText1: TStaticText;
    tmrUpdate: TTimer;
    ggLock: TGauge;
    StaticText2: TStaticText;
    StaticText3: TStaticText;
    mskFrequency: TMaskEdit;
    StaticText4: TStaticText;
    stDeviation: TStaticText;
    txtSignalPower: TStaticText;
    txtSignalState: TStaticText;
    StaticText7: TStaticText;
    stDeviationSymbolrate: TStaticText;
    txtNoiseIndicator: TStaticText;
    StaticText27: TStaticText;
    txtMs: TStaticText;
    txtPacketCount: TStaticText;
    StaticText28: TStaticText;
    StaticText29: TStaticText;
    StaticText30: TStaticText;
    txtOvertaken: TStaticText;
    StaticText5: TStaticText;
    StaticText6: TStaticText;
    StaticText8: TStaticText;
    mskVideoPid: TMaskEdit;
    mskAudioPid: TMaskEdit;
    StaticText9: TStaticText;
    StaticText10: TStaticText;
    pnlVideo: TPanel;
    StaticText11: TStaticText;
    cmbSatellite: TComboBox;
    StaticText16: TStaticText;
    lblPmtPid: TStaticText;
    mskPmtPid: TMaskEdit;
    chkSendAll: TCheckBox;
    StaticText12: TStaticText;
    txtSyncOffset: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnClearClick(Sender: TObject);
    procedure mskPidChange(Sender: TObject);
    procedure mskPidKeyPress(Sender: TObject; var Key: Char);
    procedure FormDestroy(Sender: TObject);
    procedure btnOptionSatClick(Sender: TObject);
    procedure chkSendAllClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;


var
  frmMain: TfrmMain;

implementation

{$R *.DFM}

uses
  SyncObjs,
  Windows, Messages, SysUtils, Graphics, Dialogs,
  DvbStreamBuffering,
  DvbDirectShow2,
  FlexCopInterface, FlexCopGeneral, FlexCopI2c, Stv0299bRegisters,
  FlexCopIoControl,
  FlexCopRegisters;


const
  BufferTime     = 500;
  CBufferPackets = 512;
  CPacketSize    = 188;
  CBufferSize    = CBufferPackets * CPacketSize;

type
  TBufferArray = array[0..CPacketSize-1] of Byte;

  TDataThread = class(TThread)
  private
    { Private declarations }
    FStreamData   : array[0..CBufferPackets] of TBufferArray; // Note: One too many because of synchronization
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
    FIrqId        : Dword;             // IRQ identifier for FIFO data
    FPacketBuffers: Word;              // Number of buffers used
    procedure DecodeTransportStreamData;
  protected
    procedure Execute; override;
  end;

var
  Timing : Dword;
  Counts : Dword;
  SynthesizerAddress: Byte;

  ThreadHandle: THandle;                         // Handle to driver in thread
  PacketThread: TDataThread;                     // Thread handling packets

  CardHandle  : THandle;
  CardNoError : Boolean;

  CardInUse   : Integer;

  // Debug
  PacketBufferCount  : Word;                     // Number of packet buffers received
  PacketPreviousCount: Word;                     // <PacketBufferCount> previous run
  PacketPreviousTimer: Dword;                    // Time of previous <PacketBufferCount>
  Overtaken          : Word;                     // Number of times we run 'behind', eg. NOT waiting for data
  LSyncOffset        : Word;

  NewChannelSet       : Boolean;                 // Set to True when a new channle has been set
                                                 // Used for clearing DVB tables

  PacketError         : Word;
  Buffers             : Integer;                 // Actual number of buffers used
  IsSlave             : Boolean;

  ShowVideo           : Boolean;                           // True shows video
  PidPmt   : Integer;
  PidVideo : Integer;
  PidAudio : Integer;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Decode transport stream. We simply call the filter procedures
            which have been set.
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDataThread.DecodeTransportStreamData;
var
  StreamPacket: Word;
  Pid: Word;
  ToSend: Boolean;
begin
  for StreamPacket := 0 to CBufferPackets - 1 do
  begin
    // One of the few checks at this point is the synchronization byte which must be correct
    if FStreamData[StreamPacket][0] = $47 then
    begin
      Pid := ((FStreamData[StreamPacket][1] and $1F) shl 8) or FStreamData[StreamPacket][2];
      ToSend := False;
      if Pid <> 0 then
      begin
        if (Pid = PidPmt) then
          ToSend := True;
        if (Pid = PidVideo) then
          ToSend := True;
        if (Pid = PidAudio) then
          ToSend := True;
      end
      else
        ToSend := True;
      if ToSend then
      begin
        try
          DirectShowUniversalSourceSendData(@FStreamData[StreamPacket], CPacketSize);
        except
        end;
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : 
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Buffer      : TFlexCopFifoTransferBuffer;
  SkipBuffer  : Byte;                            // Forces buffers to skip after first setup
  LastRead    : Dword;
  IrqHandling : TFlexCopIrqTransferBuffer;       // Subbuffer 0 interrupts
  SyncRequired: Integer;                         // Counts invalid sync at start buffer
  SyncOffset  : Integer;                         // Offset for correction
  SyncData    : TBufferArray;                    // Remainder data previous data block
begin
  // Debug
  PacketBufferCount :=0;

  HasStopped     := False;
  ProgrammedStop := False;
  // The driver must be running and must be in use
  if FlexCopGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't then only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls eg. 'generate manual notification'
  // which might be needed in a deadlock situation.
  ThreadHandle := FlexCopCreateFile(CardInUse);

  // Wait for application to have started and such
  repeat
   Sleep(100);
  until NewChannelSet or Terminated;
  // Dma 1, subbuffer 0 interrrupts are used
  IrqHandling.Identifier := FIrqId;
  FlexCopReadFromIrqHandling(ThreadHandle, IrqHandling);
  if not IrqHandling.Information.IrqBufferingIsActive then
    Terminate;
  LastRead := $FFFF;
  SkipBuffer := 0;
  SyncRequired := 0;
  SyncOffset := 0;
  LSyncOffset := SyncOffset;
  Buffer.TransferAddress[0] := @FStreamData[0][SyncOffset];
  try
    if not Terminated then
    repeat
      // Some special handling required when a new channel has been set
      if NewChannelSet then
      begin
        PacketError  := 0;
        Overtaken    := 0;
        NewChannelSet := False;
      end;
      if not Terminated then
      begin
        FlexCopReadFromIrqHandling(ThreadHandle, IrqHandling);
        while (not Terminated) and
              (IrqHandling.Information.FifoBufferPreviousIndex = LastRead) do
        begin
          Sleep(1);
          FlexCopReadFromIrqHandling(ThreadHandle, IrqHandling);
        end;
        if PacketBufferCount = $FFFF then
          PacketBufferCount := 0
        else
          Inc(PacketBufferCount);
        Inc(LastRead);
        if LastRead > IrqHandling.Information.FifoBufferLastIndex then
          LastRead := IrqHandling.Information.FifoBufferFirstIndex;
        // Read buffer
        Buffer.TransferLength := CBufferSize;
        Buffer.Identifier := LastRead;
        FlexCopReadFromFifo(ThreadHandle, Buffer);
        // Check first packet (the offset should have made the second packet correct)
        if (FStreamData[1][0] <> $47)  then
          Inc(SyncRequired)
        else
          if SyncRequired > 0 then
            Dec(SyncRequired);
        if SyncRequired > 50 then
        begin
          // Search for sync
          SyncOffset := 0;
          while (SyncOffset < CPacketSize) and (FStreamData[0][SyncOffset] <> $47) do
            Inc(SyncOffset);
          // Not found, default to start
          // Otherwise the correction is related to the packet size because
          // the correct data is to be placed in the second packet of
          // StreamDataBuffer
          if (SyncOffset = 0) or (SyncOffset = CPacketSize) then
            SyncOffset := 0
          else
            SyncOffset := CPacketSize - SyncOffset;
          Buffer.TransferAddress[0] := @FStreamData[0][SyncOffset];
          SyncRequired := 0;
          SkipBuffer := 1;
          LSyncOffset := SyncOffset;
        end
        else
        begin
          // We have to store the data at the end and use it for the
          // next block of data
          if SyncOffset <> 0 then
          begin
            // Copy remainder of previous block
            CopyMemory(@FStreamData, @SyncData, SyncOffset);
            // Copy leftover of current block ('one too many' packet)
            CopyMemory(@SyncData, @FStreamData[CBufferPackets], SyncOffset);
          end;
        end;

        if SkipBuffer = 0 then
        begin
          // We would normally only send the minimum required PID data
          if Assigned(frmMain) and
             ShowVideo then
          begin
            if frmMain.chkSendAll.Checked then
              DirectShowUniversalSourceSendData(@FStreamData, SizeOf(FStreamData))
            else
              DecodeTransportStreamData;
          end;
        end
        else
          Dec(SkipBuffer);
      end;
    until ProgrammedStop or Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
      ShowMessage('Unexpected termination of packets handler.');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  ErrorMessage: string;
begin
  // Check if the driver is present and/or multiple cards are present
  case FlexCopGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No FLEXCOP driver and/or card detected.');
  end;

  // We use the tag for the returned identifier
  if not DirectShowStart(frmMain.pnlVideo.Handle, 'VIDEO.GRF', False, False, False, ErrorMessage, INVALID_HANDLE_VALUE) then
    frmMain.mmoDriver.Lines.Add('DirectShow error.')
  else
  begin
    frmMain.MskPidChange(nil);  // Send PID info to DirectShow
    ShowVideo := True;
  end;

  if IsSlave then
  begin
    cmbSatellite.Enabled  := False;
    mskFrequency.Enabled  := False;
    mskSymbolrate.Enabled := False;
    frmMain.mmoDriver.Lines.Add('Set to operate as slave.')
  end
  else
    frmMain.mmoDriver.Lines.Add('Operating as master.');
  if not Assigned(PacketThread) then
    frmMain.mmoDriver.Lines.Add('Buffering setup error.')
  else
    frmMain.mmoDriver.Lines.Add(format('Using %d buffers (appr. %d ms).', [Buffers, Buffers*20]));
  NewChannelSet := True;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Form destruction.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  // Note: We MUST make sure that <DSSendDataDirect> is never called when DirectShow
  //       has stopped, therefore we inform the thread
  ShowVideo := False;
  Sleep(1);

  DirectShowStop;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  if not FlexCopQpskSetSymbolRate(CardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  StrengthDecibel   : Double;
  StrengthPercentage: Double;
  DeltaTime : Dword;
  DeltaCount: Dword;
  Ticks     : Dword;
  IrqHandling: TFlexCopIrqTransferBuffer;       // Subbuffer 0 interrupts
begin
  // Calculate time for each packetbuffer
  Ticks := GetTickCount;
  DeltaTime := Abs(Ticks - PacketPreviousTimer);
  PacketPreviousTimer := Ticks;
  if PacketBufferCount <> PacketPreviousCount then
  begin
    if PacketBufferCount > PacketPreviousCount then
      DeltaCount := PacketBufferCount - PacketPreviousCount
    else
      DeltaCount := PacketPreviousCount - PacketBufferCount;
    txtMs.Caption := IntToStr(DeltaTime div DeltaCount);
    PacketPreviousCount := PacketBufferCount;
  end;
  txtPacketCount.Caption := IntToStr(PacketBufferCount);

  IrqHandling.Identifier := CFlexCopIrqDma10Irq;
  FlexCopReadFromIrqHandling(CardHandle, IrqHandling);
  txtOvertaken.Caption   := IntToStr(IrqHandling.Information.FifoOverflows);
  txtSyncOffset.Caption   := IntToStr(LSyncOffset);

  if FlexCopQpskGetLockDetectorPercentage(CardHandle, Percentage) then
    ggLock.Progress := Percentage;
  if FlexCopQpskGetCarrierDeviation(CardHandle, Deviation) then
    stDeviation.Caption := format('%d', [Deviation]);
  if FlexCopQpskGetSymbolRateDeviation(CardHandle, Deviation) then
    stDeviationSymbolRate.Caption := format('%d', [Deviation]);
  if FlexCopQpskGetSignalPower(CardHandle, StrengthPercentage, StrengthDecibel) then
  begin
    txtSignalPower.Caption := format('%f%% (%f dB)', [StrengthPercentage, StrengthDecibel]);
  end;
  if FlexCopReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
  begin
    if (LockState and $80) <> 0 then
      StatusStr := 'CARRIER'
    else
      StatusStr := 'NO CARRIER';
    if (LockState and $10) <> 0 then
      StatusStr := StatusStr + ' - VITERBI'
    else
      StatusStr := StatusStr + ' - NO VITERBI';
    if (LockState and $08) <> 0 then
      StatusStr := StatusStr + ' - SYNC'
    else
      StatusStr := StatusStr + ' - NO SYNC';
    if (LockState and $98) = $98 then
      StatusStr := StatusStr + ' - LOCK'
    else
      StatusStr := StatusStr + ' - NO LOCK';
    txtSignalState.Caption := StatusStr;
  end;
  if FlexCopQpskGetNoiseIndicator(CardHandle, Noise) then
  begin
    txtNoiseIndicator.Caption := format('%f dB', [Noise]);
  end;

end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error:     Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert frequency.');
    Exit;
  end;
  if not FlexCopFrequencyLNB(CardHandle, Frequency, 9750000, 10600000, 0, SynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;

procedure TfrmMain.mskPidKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskPidChange(nil);
end;

procedure TfrmMain.btnClearClick(Sender: TObject);
begin
  NewChannelSet := True;
  mmoDriver.Clear;
end;

procedure TfrmMain.mskPidChange(Sender: TObject);
var
  Error    : Integer;
begin
  Val(mskPmtPid.EditText, PidPmt, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert PMT PID.');
    Exit;
  end;
  Val(mskVideoPid.EditText, PidVideo, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Video PID.');
    Exit;
  end;
  Val(mskAudioPid.EditText, PidAudio, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Audio PID.');
    Exit;
  end;

  DirectShowMpeg2demultiplexerSetNewPids(PidVideo, PidAudio);
end;


procedure TfrmMain.btnOptionSatClick(Sender: TObject);
var
  AString   : ShortString;
begin
  case cmbSatellite.ItemIndex of
    0: AString := '1 11700 V 09750 t v W15 [E0 10 38 F0] W15 t';
    1: AString := '1 99999 V 10600 t v W15 [E0 10 38 F1] W15 T';
    2: Astring := '1 11700 H 09750 t V W15 [E0 10 38 F2] W15 t';
    3: AString := '1 99999 H 10600 t V W15 [E0 10 38 F3] W15 T';
    4: AString := '2 11700 V 09750 t v W15 [E0 10 38 F4] W15 t';
    5: AString := '2 99999 V 10600 t v W15 [E0 10 38 F5] W15 T';
    6: AString := '2 11700 H 09750 t V W15 [E0 10 38 F6] W15 t';
    7: AString := '2 99999 H 10600 t V W15 [E0 10 38 F7] W15 T';
    else
      Exit;
  end;
  if not FlexCopDiSEqCCommandVdr(CardHandle, AString) then
    mmoDriver.Lines.Add('DiSEqC command error')
  else
    mmoDriver.Lines.Add(format('DiSEqC command success setting %d.', [cmbSatellite.ItemIndex]));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  IrqId: Dword;
begin
  SynthesizerAddress := 1;
  NewChannelSet := False;
  ShowVideo     := False;
  Timing        := 0;
  Counts        := 0;
  PacketThread  := nil;
  CardHandle    := INVALID_HANDLE_VALUE;
  CardNoError   := False;
  CardInUse     := -1;
  if FlexCopGetNumberOfCards <> 0 then
  begin
    CardHandle := FlexCopCreateFile(CardInUse);
    CardInUse  := FlexCopGetCardOfHandle(CardHandle);
  end;
  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    Buffers := BufferTime;
    if ParamCount = 1 then
      IsSlave := True
    else
      IsSlave := False;  
    if not DvbSetupBuffering(CardHandle, True, 0, 0, @IsSlave,
      @Buffers, @IrqId, 0) then
      Exit;
  end;
  // Start the thread which receives notifications we setup next
  PacketThread := TDataThread.Create(True);
  PacketThread.FIrqId         := IrqId;
  PacketThread.FPacketBuffers := Buffers;
  PacketThread.Priority       := tpTimeCritical;
  PacketThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
var
  Channel   : Dword;
  Dummy     : Dword;
  FifoBuffer: TFlexCopFifoTransferBuffer;
begin
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    FlexCopGenerateManualNotification(ThreadHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;
  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    // Disable all interrupts
    if not IsSlave then
    begin
      FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0);
      // Clear pending interrupt(s)
      FlexCopReadFromFlexCopRegister(CardHandle, CFlexCopEnable, Dummy);
      // Disable power to LNB
      FlexCopDisableLNB(CardHandle);
      // Pseudo reset
      FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopSpecials, CFlexCopSpecialReset);
      // Release FIFO allocated memory
      FifoBuffer.Identifier := -1;
      FlexCopReleaseFifo(CardHandle, FifoBuffer);
      // Release DMA allocated memory
      Channel := 0;
      while FlexCopReleaseDma(CardHandle, Channel) do
        Inc(Channel);
    end;    
    FlexCopCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;

end;


procedure TfrmMain.chkSendAllClick(Sender: TObject);
begin
  lblPmtPid.Visible := not chkSendAll.Checked;
  mskPmtPid.Visible := not chkSendAll.Checked;
end;

initialization
  Initialize;


finalization
  Finalize;
end.
