{******************************************************************************}
{ FileName............: GettingPacketDataUnit001                               }
{ Project.............: FLEXCOP                                                }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Getting packet data.                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050409 - Initial WDM release                                       }
{******************************************************************************}
unit GettingPacketDataUnit001;

interface

uses
  SyncObjs,
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FlexCopInterface, FlexCopGeneral, FlexCopI2c, Stv0299bRegisters,
  ExtCtrls, Mask, Gauges,
  Dvb,
  FlexCopIoControl,
  FlexCopRegisters;

type
  TDataThread = class(TThread)
  private
    { Private declarations }
  protected
    constructor Create;
    procedure Execute; override;
  end;

  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    mskSymbolRate: TMaskEdit;
    StaticText1: TStaticText;
    tmrUpdate: TTimer;
    ggLock: TGauge;
    StaticText2: TStaticText;
    StaticText3: TStaticText;
    mskFrequency: TMaskEdit;
    StaticText4: TStaticText;
    stDeviation: TStaticText;
    StaticText5: TStaticText;
    StaticText6: TStaticText;
    StaticText7: TStaticText;
    stDeviationSymbolrate: TStaticText;
    btnBeforeLock: TButton;
    btnAfterLock: TButton;
    rgLNB: TRadioGroup;
    StaticText8: TStaticText;
    rgInversion: TRadioGroup;
    StaticText15: TStaticText;
    Button1: TButton;
    lblIrq1: TLabel;
    lblIrq2: TLabel;
    Button2: TButton;
    lblDma1: TLabel;
    lblDma2: TLabel;
    cmbSatellite: TComboBox;
    StaticText16: TStaticText;
    StaticText20: TStaticText;
    StaticText21: TStaticText;
    lblRotates: TLabel;
    lblLost: TLabel;
    StaticText9: TStaticText;
    StaticText10: TStaticText;
    btnThread: TButton;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnBeforeLockClick(Sender: TObject);
    procedure btnAfterLockClick(Sender: TObject);
    procedure rgLNBClick(Sender: TObject);
    procedure btnResetClick(Sender: TObject);
    procedure rgInversionClick(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure Button2Click(Sender: TObject);
    procedure btnOptionSatClick(Sender: TObject);
    procedure btnThreadClick(Sender: TObject);
  private
    { Private declarations }
    FCardNoError: Boolean;
    FThread: TDataThread;
    FSynthesizerAddress: Byte;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;
  Timing : Dword;
  Counts : Dword;
  ThreadCounts: Dword;
  OddCounts   : Dword;
  EvenCounts  : Dword;
  HandleCard  : THandle;
  FifoBuffer  : TFlexCopFifoTransferBuffer;      // Data buffering
  IrqHandling0: TFlexCopIrqTransferBuffer;       // Subbuffer 0 interrupts
  IrqHandling1: TFlexCopIrqTransferBuffer;       // Subbuffer 1 interrupts
  IrqsActive  : Boolean;                         // Valid Irq buffers
  BufferSize  : Integer;                         // Actual buffer size
  Addresses   : array[0..30] of Dword;            // Recorded addresses
  AddressIndex: Integer;                         // Recorded addresses max
  Buffer1    : TFlexCopDmaBuffer;
  Buffer2    : TFlexCopDmaBuffer;

  LastIrqs1        : Dword;
  LastIrqs1Recorded: Dword;
  LastIrqs1Timing  : Dword;
  LastIrqs1Delta   : Integer;

  LastIrqs2        : Dword;
  LastIrqs2Recorded: Dword;
  LastIrqs2Timing  : Dword;
  LastIrqs2Delta   : Integer;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  LCardHandle: THandle;                                    // Handle to driver
  Buffer     : TFlexCopFifoTransferBuffer;
  DvbData    : array[0..CDvbPacketVSync * CDvbPacketHSync-1] of Byte;
  OutFile    : TFileStream;
  AutoEnd    : Dword;
  LastRead   : Dword;
begin

  // The driver must be running and must be in use
  if FlexCopGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls (eg. 'generate manual notification'
  // which migth be needed in a deadlock situation.
  LCardHandle := FlexCopCreateFile(0);
  ThreadCounts := 0;
  OddCounts    := 0;
  EvenCounts   := 0;
  AutoEnd := 100;
  FlexCopReadFromIrqHandling(LCardHandle, IrqHandling0);
  LastRead := IrqHandling0.Information.FifoBufferPreviousIndex;
  Buffer.TransferAddress[0] := @DvbData;
  OutFile := TFileStream.Create('TransportStream.TS', fmCreate);
  try
    repeat
      // In this thread we wait for fifo changes and act on it (we could
      // also wait for a notification though)...
      Sleep(1);                                              // Essential!!
      FlexCopReadFromIrqHandling(LCardHandle, IrqHandling0);
      // Check for available data
      if IrqHandling0.Information.FifoBufferPreviousIndex <> LastRead then
      begin
        // Read buffer
        Buffer.TransferLength := CDvbPacketVSync * CDvbPacketSize;
        Buffer.Identifier := LastRead;
        if FlexCopReadFromFifo(LCardHandle, Buffer) then
        begin
          OutFile.Write(DvbData, Buffer.TransferLength);
          Inc(ThreadCounts);
          Dec(AutoEnd);
          if AutoEnd = 0 then
            Terminate;
          Inc(LastRead);
          if LastRead > IrqHandling0.Information.FifoBufferLastIndex then
            LastRead := IrqHandling0.Information.FifoBufferFirstIndex;
        end;
      end;
    until Terminated;
  finally
    OutFile.Free;
  end;
  FlexCopCloseHandle(LCardHandle);
  frmMain.btnThread.Visible := False;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Creation of thread.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TDataThread.Create;
begin
  // Create and run (do not suspend)
  inherited Create(False);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
begin
  Timing := 0;
  Counts := 0;
  FThread := nil;
  FCardNoError := False;
  // Check if the driver is present and/or multiple cards are present
  if FlexCopGetNumberOfCards = 0 then
    mmoDriver.Lines.Add('No FlexCop driver and/or card detected.')
  else
    mmoDriver.Lines.Add('FlexCop driver detected.');
  btnResetClick(nil);
  cmbSatellite.ItemIndex := 6;
  cmbSatellite.OnChange(nil);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  if Assigned(FThread) then
  begin
    FThread.Terminate;                                     // Mark it for termination
    if FThread.Suspended then
      FThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    FlexCopGenerateManualNotification(HandleCard);
    FThread.WaitFor;
    FreeAndNil(FThread);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    exit;
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  if not FlexCopQpskSetSymbolRate(HandleCard, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  PowerPercentage: Double;
  PowerDecibel   : Double;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  DmaStatus  : TFlexCopGetDmaStatus;
  Delta      : Integer;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    exit;
  // We can check the interrupt status if the setup was successful
  if IrqsActive then
  begin
    if FlexCopReadFromIrqHandling(HandleCard, IrqHandling0) then
    begin
      LastIrqs1 := Irqhandling0.Information.Irqs;
      Delta := Abs(Integer(LastIrqs1Timing) - Integer(GetTickCount));
      if Delta > 1000 then
      begin
        if LastIrqs1 = LastIrqs1Recorded then
          LastIrqs1Delta := 0
        else
          if LastIrqs1 < LastIrqs1Recorded then
            LastIrqs1Delta := Trunc(((LastIrqs1Recorded - LastIrqs1)*1000) / Delta)
          else
            LastIrqs1Delta := Trunc(((LastIrqs1 - LastIrqs1Recorded)*1000) / Delta);
        LastIrqs1Recorded := LastIrqs1;
        LastIrqs1Timing   := GetTickCount;
      end;
      lblIrq1.Caption := format('%d @ %d /s', [LastIrqs1, LastIrqs1Delta]);
      lblRotates.Caption := format('%d full buffers', [Irqhandling0.Information.FifoBufferCirculations]);
      lblLost.Caption := format('%d lost buffers', [Irqhandling0.Information.FifoOverflows]);
    end;

    if FlexCopReadFromIrqHandling(HandleCard, IrqHandling1) then
    begin
      LastIrqs2 := Irqhandling1.Information.Irqs;
      Delta := Abs(Integer(LastIrqs2Timing) - Integer(GetTickCount));
      if Delta > 1000 then
      begin
        if LastIrqs2 = LastIrqs2Recorded then
          LastIrqs2Delta := 0
        else
          if LastIrqs2 < LastIrqs2Recorded then
            LastIrqs2Delta := Trunc(((LastIrqs2Recorded - LastIrqs2)*1000) / Delta)
          else
            LastIrqs2Delta := Trunc(((LastIrqs2 - LastIrqs2Recorded)*1000) / Delta);
        LastIrqs2Recorded := LastIrqs2;
        LastIrqs2Timing   := GetTickCount;
      end;
      lblIrq2.Caption := format('%d == %d /s', [LastIrqs2, LastIrqs2Delta]);
    end;

    if FlexCopGetDmaStatus(HandleCard, Buffer1.Identifier, DmaStatus) then
      lblDma1.Caption := format('$%p, %d', [DmaStatus.VirtualAddress, DmaStatus.Interrupts]);
    if FlexCopGetDmaStatus(HandleCard, Buffer2.Identifier, DmaStatus) then
      lblDma2.Caption := format('$%p, %d', [DmaStatus.VirtualAddress, DmaStatus.Interrupts]);
  end;

  if FlexCopQpskGetLockDetectorPercentage(HandleCard, Percentage) then
    ggLock.Progress := Percentage;
  if FlexCopQpskGetCarrierDeviation(HandleCard, Deviation) then
    stDeviation.Caption := format('%d', [Deviation]);
  if FlexCopQpskGetSymbolRateDeviation(HandleCard, Deviation) then
    stDeviationSymbolRate.Caption := format('%d', [Deviation]);
  if FlexCopQpskGetSignalPower(HandleCard, PowerPercentage, PowerDecibel) then
  begin
    statictext5.caption := format('Power %f%% (%f dB)', [PowerPercentage, PowerDecibel]);
  end;
  if FlexCopReadFromQpsk(HandleCard, CStv0299bVStatus, LockState) then
  begin
    if (LockState and $80) <> 0 then
      StatusStr := 'CARRIER'
    else
      StatusStr := 'NO CARRIER';
    if (LockState and $10) <> 0 then
      StatusStr := StatusStr + ' - VITERBI'
    else
      StatusStr := StatusStr + ' - NO VITERBI';
    if (LockState and $08) <> 0 then
      StatusStr := StatusStr + ' - SYNC'
    else
      StatusStr := StatusStr + ' - NO SYNC';
    if (LockState and $98) = $98 then
      StatusStr := StatusStr + ' - LOCK'
    else
      StatusStr := StatusStr + ' - NO LOCK';
    statictext6.caption := format('Status $%2.2x : %s', [LockState, StatusStr]);
  end;
  if FlexCopQpskGetNoiseIndicator(HandleCard, Noise) then
  begin
    statictext8.caption := format('Noise %f dB', [Noise]);
  end;

end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error:     Integer;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert frequency.');
    Exit;
  end;
  // Befi

  if not FlexCopFrequencyLNB(HandleCard, Frequency, 9750000, 10600000, 0, FSynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;

procedure TfrmMain.btnBeforeLockClick(Sender: TObject);
var
  Data: Byte;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  if not FlexCopReadFromQpsk(HandleCard, CStv0299bCfd, Data) then
    Exit;
  Data := Data or  $80;
  FlexCopWriteToQpsk(HandleCard, CStv0299bCfd, Data);
end;

procedure TfrmMain.btnAfterLockClick(Sender: TObject);
var
  Data: Byte;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  if not FlexCopReadFromQpsk(HandleCard, CStv0299bCfd, Data) then
    Exit;
  Data := Data and $7F;
  FlexCopWriteToQpsk(HandleCard, CStv0299bCfd, Data);
end;

procedure TfrmMain.rgLNBClick(Sender: TObject);
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  case rgLNB.ItemIndex of
    0: if not FlexCopEnableLNB(HandleCard) then
         mmoDriver.Lines.Add('Could not enable LNB.');
    1: if not FlexCopDisableLNB(HandleCard) then
         mmoDriver.Lines.Add('Could not disable LNB.');
  end;
end;

procedure TfrmMain.btnResetClick(Sender: TObject);
var
  ReturnedByte : Byte;
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  FCardNoError := False;
  FSynthesizerAddress := FlexCopDetectSynthesizerAddress(HandleCard);

  if not FlexCopReadFromQpsk(HandleCard, CStv0299bId, ReturnedByte) then
  begin
    mmoDriver.Lines.Add('Could not read identification of QPSK demodulator.');
    Exit;
  end;
  if not ReturnedByte = $A1 then
  begin
    mmoDriver.Lines.Add(format('QPSK demodulator identification NOT correct ($%x).', [ReturnedByte]));
    Exit;
  end;
  if not FlexCopWriteDefaultsToQpsk(HandleCard, 0) then
  begin
    mmoDriver.Lines.Add('QPSK registers could not be set to default values.');
    Exit;
  end;
  FCardNoError := True;
  rgLNBClick(nil);
  rgInversionClick(nil);
  mskFrequencyChange(nil);
  mskSymbolrateChange(nil);
  if FCardNoError then
    tmrUpdate.Enabled := True;
end;

procedure TfrmMain.rgInversionClick(Sender: TObject);
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  case rgInversion.ItemIndex of
    0: if not FlexCopQpskSetInversion(HandleCard, True) then
         mmoDriver.Lines.Add('Could not set inversion.');
    1: if not FlexCopQpskSetInversion(HandleCard, False) then
         mmoDriver.Lines.Add('Could not reset inversion.');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>    Handle to driver
            <DmaChannel2>   False for DMA1
                            True  for DMA2
            <ActivateDma>   True if DMA also to be activated (in address)
            <Packets>       Number of packets to use for size
            <SubBuffer0>    Pointer to physical address DMA subbuffer 0
            <SubBuffer1>    Pointer to physical address DMA subbuffer 0
            <PacketIrqs>    True if packet interrupts are to be enabled
  Returns : <Result>        True if success

  Descript: Setup FlexCop registers
  Notes   :
 ------------------------------------------------------------------------------}
function SetupFlexCopAddress(CardHandle: THandle; DmaChannel2: Boolean; ActivateDma: Boolean;
           Packets: Word; SubBuffer0: Cardinal; SubBuffer1: Cardinal; PacketIrqs: Boolean): Boolean;
var
  Dma        : Dword;
  Address    : Dword;
  PacketSized: Integer;
  SetSize    : Dword;
begin
  Result      := False;
  PacketSized := Packets * CDvbPacketSize;
  if ActivateDma then
    Dma := CFlexCopSubBufferEnableIrq
  else
    Dma := 0;

  // Fortunately our allocated memory is always dword alligned ....
  // Buffer 1 address
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer0Address
  else
    Address := CFlexCopDma1SubBuffer0Address;
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer0 or Dma) then
    Exit;

  // Sub buffer 1 is either enabled or disbaled (nothing to do with actual DMA enabling)  
  if ActivateDma then
    Dma := CFlexCopSubBufferEnableSubBuffer1
  else
    Dma := 0;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer1Address
  else
    Address := CFlexCopDma1SubBuffer1Address;
  // Buffer 2 address
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer1 or Dma) then
    Exit;

  // Buffer size
  // We have allocated memory which is a multiple of the DMA block size.
  // For convenience we adjust this to the even packet size.
  SetSize := PacketSized;
  SetSize := SetSize shr 2;                              // Size if 4-byte chunks
  SetSize := SetSize shl 8;                              // Shift size
  if not PacketIrqs then
    SetSize := SetSize or CFlexCopSubBufferDisablePacketIrq;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBufferSize
  else
    Address := CFlexCopDma1SubBufferSize;
  if not FlexCopWriteToFlexCopRegister(CardHandle, Address, SetSize) then
    Exit;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>     Handle to driver
            <IsSlave>        True if setup as slave
            <PacketBuffers>  Number of buffers to allocate
            <PacketIrqs>     True if packet interrupts to be enabled
  Returns : <Result>         True if success
            <PacketBuffers>  Actual buffers allocated
            <BufferId>       Buffer identifier (DMA) for data stored into

  Descript: Setup buffering
  Notes   : We don't remove allocated DMA/FIFO's and such since
            it is implicitly done at the start here and in the
            termination of the application
 ------------------------------------------------------------------------------}
function BufferingSetup(CardHandle: THandle; IsSlave: Boolean;
  var PacketBuffers: Word; var BufferId: Dword; PacketIrqs: Boolean): Boolean;
var
  Channel    : Dword;
  Buffer3    : TFlexCopDmaBuffer;
  Buffer4    : TFlexCopDmaBuffer;

begin
  Result     := False;
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;

  // Setup DMA
  //
  // We need to setup several buffers.
  // Buffer 1:  FlexCop target buffer for satellite data (subbuffer 0) first DMA
  // Buffer 2:  FlexCop target buffer for satellite data (subbuffer 1) first DMA
  // Buffer 3:  Dummy buffer for second DMA
  // Buffer 4:  Dummy buffer for second DMA
  //
  // The FlexCop issues an interrupt when one of it's buffers is filled (in our
  // case this buffer is buffer 1 or 2). The driver (if properly set up) then
  // copies this data to somewere in buffer 3.
  // Typically the FlexCop issues two interrupts. One when it's first subbuffer
  // has been filled, and another when it's second subbuffer has been filled.
  // Using the 'sharing' mechanism of the FIFO we can make sure that all data
  // is placed on our common buffer 3.
  //
  // We are dealing with DMA 1 only here (DMA2 is not used). To recap, the
  // following registers/settings are to be set up:
  // DMA1 - generates two distinguishes interrupts: One for subbuffer 0, and
  //        another for subbuffer 1
  //        CFlexCopIrqDma10                : DMA1, subbuffer 0 interrupt
  //        CFlexCopIrqDma11                : DMA1, subbuffer 1 interrupt
  //        CFlexCopEnableDma10             : DMA1, subbuffer 0 interrupt enable
  //        CFlexCopEnableDma11             : DMA1, subbuffer 1 interrupt enable
  //        CFlexCopDma1SubBuffer0Address   : DMA1, subbuffer 0 target address
  //        CFlexCopDma1SubBufferSize       : DMA1, subbuffer size
  //        CFlexCopDma1SubBuffer1Address   : DMA1, subbuffer 1 target address

  // First disable all possible interrupts and such
  if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0) then
    Exit;

  // Release any memory already allocated
  Channel := 0;
  while FlexCopReleaseDma(CardHandle, Channel) do
    Inc(Channel);
  // Release all FIFO's
  FifoBuffer.Identifier := -1;
  FlexCopReleaseFifo(CardHandle, FifoBuffer);

  // Allocate buffer 1
  if not FlexCopAllocateDma(CardHandle, (((CDvbPacketBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer1) then
    Exit;
  // Allocate buffer 2
  if not FlexCopAllocateDma(CardHandle, (((CDvbPacketBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer2) then
    Exit;
  // Dummy buffers so we have at least a valid buffer, which we use for DMA2
  // Allocate buffer 3
  if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer3) then
    Exit;
  // Allocate buffer 4
  if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer4) then
    Exit;

  // Setup initial addresses/size
  BufferSize := CDvbPacketVSync * CDvbPacketSize;
  if not SetupFlexCopAddress(CardHandle, False, False, CDvbPacketVSync,
    Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;
  // Also do this for DMA2 although we don't use DMA2
  // (just to make sure DMA addresses are valid)
  if not SetupFlexCopAddress(CardHandle, True, False, DMASIZE div CDvbPacketSize,
    Buffer3.PhysicalAddress.LowPart, Buffer4.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;

  // Setup Fifo. This allocates the number of buffers we want.
  with FifoBuffer do
  begin
    NumberOfBuffers := PacketBuffers;
    // Make sure all unused items are reset
    for Channel := Low(TransferAddress) to High(TransferAddress) do
      TransferAddress[Channel] := nil;
    // Note: The VIRTUAL address !
    // Fill in for subbuffer 0 and 1
    TransferAddress[CFlexCopIrqDma10Irq] := Pointer(Buffer1.VirtualAddress);
    TransferAddress[CFlexCopIrqDma11Irq] := Pointer(Buffer2.VirtualAddress);
    TransferLength := BufferSize;
  end;
  if not FlexCopAllocateFifo(CardHandle, FifoBuffer) then
    Exit;
    
  // At this point all buffers has been setup (allocated):
  //  Buffer 1 and 2 (DMA mechanism, so contiguous memory)

  // Enable interrupt mechanism of the driver
  // When this is set the actual buffering of data (to the FIFO) is activated
  // Setup for subbuffer 0 (DMA1 - subbuffer 0)
  with IrqHandling0.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := False;
    UseSignaling         := False;
    UseFifo              := True;
    FifoBufferFirstIndex := FifoBuffer.Identifier;
    FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
  end;
  IrqHandling0.Identifier := CFlexCopIrqDma10Irq;
  FlexCopWriteToIrqHandling(CardHandle, IrqHandling0);

  // Setup for subbuffer 1 (DMA1 - subbuffer 1)
  // Subbuffer 1 uses the same fifo buffers as subbuffer 0
  with IrqHandling1.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := False;
    UseSignaling         := False;
    UseFifo              := True;
    FifoBufferFirstIndex := FifoBuffer.Identifier;
    FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
  end;
  IrqHandling1.Identifier := CFlexCopIrqDma11Irq;
  FlexCopWriteToIrqHandling(CardHandle, IrqHandling1);

  // Now we have to enable DMA of the FlexCop. We already have setup the address
  // and such.

  // Enable subbuffer 0/1 DMA
  // Now we have to enable the reception of data
  // First the DMA bits in the address of the subbuffers
  if not SetupFlexCopAddress(CardHandle, False, True, CDvbPacketVSync,
    Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;

  // Enable reception of data
  FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable,
    CFlexCopEnableDma10 or CFlexCopEnableDma11 or CFlexCopEnableReceive);

  Result := True;
end;


procedure TfrmMain.Button1Click(Sender: TObject);
var
  Packets : Word;
  BufferId: Dword;
begin
  Packets := 10;
  IrqsActive := BufferingSetup(HandleCard, False, Packets, BufferId, True);
  if IrqsActive then
  begin
    mmoDriver.Lines.Add('Buffering setup successful.');
    btnThread.Visible := True;
  end
  else
  begin
    mmoDriver.Lines.Add('Buffering setup did not succeed.');
    btnThread.Visible := False;
  end;
end;




{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  Dummy: Dword;
begin
  HandleCard := INVALID_HANDLE_VALUE;
  IrqsActive := False;
  // Get handle to driver (card 0)
  if FlexCopGetNumberOfCards <> 0 then
    HandleCard := FlexCopCreateFile(0);
  if HandleCard <> INVALID_HANDLE_VALUE then
  begin
    // Wake up
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn);
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn);
    Sleep(100);
    // Clear pending interrupt(s)
    FlexCopReadFromFlexCopRegister(HandleCard, CFlexCopEnable, Dummy);
    // Disable all interrupts
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopEnable, 0);
    // Pseudo reset
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopSpecials, CFlexCopSpecialReset);
    Sleep(1);
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, 0);
    Sleep(1);
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, CFlexCopBoardReset);
    Sleep(1);

    // Wake up
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn);
    if not FlexCopReadFromFlexCopRegister(HandleCard, CFlexCopPowerAndRevision, Dummy) then
      Exit;

    FlexCopWriteDefaultsToQpsk(HandleCard, 0);

    FlexCopRamDetectSize(HandleCard);
    FlexCopSelect(HandleCard, CFlexCopMemorySelectMedia, CFlexCopMemorySelect1);     // Required


    // Enable null filter
// -   FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopEnable, CFlexCopEnableNull);

// -   FlexCopSelect(HandleCard, CFlexCopMemorySelectNet,   CFlexCopMemorySelect1);
// -  FlexCopSelect(HandleCard, CFlexCopMemorySelectCaIn,  CFlexCopMemorySelect2);
// -  FlexCopSelect(HandleCard, CFlexCopMemorySelectCaOut, CFlexCopMemorySelect2);

    // Set hardware PIDs
    // Stream 1
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidStream12, $0000FFFF, $00001FFF);
    // Stream 2
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidStream12, $FFFF0000, $1FFF0000);
    // PCR
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidPcrPmt,   $0000FFFF, $00001FFF);
    // PMT
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidPcrPmt,   $FFFF0000, $1FFF0000);
    // EMM
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidEmmEcm,   $0000FFFF, $00001FFF);
    // ECM
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopPidHardwarePidEmmEcm,   $FFFF0000, $1FFF0000);
    // Group PID
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopHardwarePidMask,   $FFFF0000, 0);
    // Group mask ($00..$1F)
//    FlexCopWriteWithMaskToFlexCopRegister(HandleCard, CFlexCopHardwarePidMask,   $0000FFFF, $1FE00000);
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
var
  Channel: Integer;
  Dummy  : Dword;
begin
  if HandleCard <> INVALID_HANDLE_VALUE then
  begin
    // Disable all interrupts
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopEnable, 0);
    // Clear pending interrupt(s)
    FlexCopReadFromFlexCopRegister(HandleCard, CFlexCopEnable, Dummy);
    // Remove DMA addresses
    SetupFlexCopAddress(HandleCard, False, False, 0, 0, 0, False);
    SetupFlexCopAddress(HandleCard, True,  False, 0, 0, 0, False);
    // Disable power to LNB
    FlexCopDisableLNB(HandleCard);
    // Pseudo reset
    FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopSpecials, CFlexCopSpecialReset);
    // Release FIFO allocated memory
    FifoBuffer.Identifier := -1;
    FlexCopReleaseFifo(HandleCard, FifoBuffer);
    // Release DMA allocated memory
    Channel := 0;
    while FlexCopReleaseDma(HandleCard, Channel) do
      Inc(Channel);
    FlexCopCloseHandle(HandleCard);;
  end;
end;


procedure TfrmMain.Button2Click(Sender: TObject);
begin
  if HandleCard = INVALID_HANDLE_VALUE then
    Exit;
  if not FlexCopWriteToFlexCopRegister(HandleCard, CFlexCopEnable, 0) then
    Exit;
  btnThread.Visible := False;  
end;

procedure TfrmMain.btnOptionSatClick(Sender: TObject);
var
  AString   : ShortString;
begin
  case cmbSatellite.ItemIndex of
    0: AString := '1 11700 V 09750 t v W15 [E0 10 38 F0] W15 t';
    1: AString := '1 99999 V 10600 t v W15 [E0 10 38 F1] W15 T';
    2: Astring := '1 11700 H 09750 t V W15 [E0 10 38 F2] W15 t';
    3: AString := '1 99999 H 10600 t V W15 [E0 10 38 F3] W15 T';
    4: AString := '2 11700 V 09750 t v W15 [E0 10 38 F4] W15 t';
    5: AString := '2 99999 V 10600 t v W15 [E0 10 38 F5] W15 T';
    6: AString := '2 11700 H 09750 t V W15 [E0 10 38 F6] W15 t';
    7: AString := '2 99999 H 10600 t V W15 [E0 10 38 F7] W15 T';
    else
      Exit;
  end;
  if not FlexCopDiSEqCCommandVdr(HandleCard, AString) then
    mmoDriver.Lines.Add('DiSEqC command error.')
  else
    mmoDriver.Lines.Add(format('DiSEqC command success setting %d.', [cmbSatellite.ItemIndex]));
end;


procedure TfrmMain.btnThreadClick(Sender: TObject);
begin
  FThread := TDataThread.Create;
  mmoDriver.Lines.Add('Capturing started, wait for button to disappear.');
end;

initialization
  Initialize;

finalization
  Finalize;

end.