{******************************************************************************}
{ FileName............: CheckingHardwareUnit001                                }
{ Project.............: FLEXCOP                                                }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Checking the hardware.                                                      }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040730 - Initial WDM release                                       }
{******************************************************************************}
unit CheckingHardwareUnit001;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FlexCopInterface, FlexCopRegisters, FlexCopGeneral, FlexCopI2c, ExtCtrls;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.DFM}


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  CardHandle         : THandle;
  MajorVersion       : Word;
  MinorVersion       : Word;
  Build              : Dword;
  DeviceName         : string;
  Data               : Dword;
  MacAddress         : Int64;
  TunerType          : Longword;
  ByteArray          : array of Byte;
  NewStr             : string;
  Loop               : Integer;
  Identification     : Word;
  AByte              : Byte;
  SynthesizerAddress : Byte;
  Size               : Word;
  CheckByte          : Byte;
  Error              : Boolean;
begin
  // Check if the driver is present and/or multiple cards are present
  case FlexCopGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No FLEXCOP driver and/or card detected.');
    1:   mmoDriver.Lines.Add('One FLEXCOP card detected.');
    else mmoDriver.Lines.Add(format('%d FLEXCOP cards detected.', [FlexCopGetNumberOfCards]));
  end;
  // If any card detected then display additional information
  if FlexCopGetNumberOfCards <> 0 then
  begin
    // At this time we don't have a link to the driver itself. We have to create this.
    // We use the first available card (-1==auto) there is.
    CardHandle := FlexCopCreateFile(-1);
    try
      // Now with the handle we can actually access the driver.
      // Start with getting the version information of the driver.
      if FlexCopGetDriverVersion(CardHandle, MajorVersion, MinorVersion, Build, DeviceName) then
        mmoDriver.Lines.Add(format('Driver "%s", Version %d.%2.2d, Build %8.8x.', [DeviceName, MajorVersion, MinorVersion, Build]));

      // Pseudo reset
      FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopSpecials, CFlexCopSpecialReset);
      Sleep(1);
      FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, 0);
      Sleep(1);
      FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, CFlexCopBoardReset);
      Sleep(1);

      FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn);
      if FlexCopReadFromFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, Data) then
      begin
        mmoDriver.Lines.Add(format('FlexCop power/revision register: $%8.8x', [Data]));
        if (Data and CFlexCopTunerPowerMask) = CFlexCopTunerPowerOn then
          mmoDriver.Lines.Add('Tuner/LNB power is on')
        else
          mmoDriver.Lines.Add('Tuner/LNB power is off');
        if (Data and CFlexCopSystemPowerMask) = CFlexCopSystemPowerOn then
          mmoDriver.Lines.Add('System power is on')
        else
          mmoDriver.Lines.Add('System power is off');
        if (Data and CFlexCopLnbPolarityMask) = CFlexCopLnbPolarityHorizontal then
          mmoDriver.Lines.Add('LNB horizontal polarity set')
        else
          mmoDriver.Lines.Add('LNB horizontal vertical set');
        if (Data and CFlexCopWakeUpMask) = CFlexCopWakeUpOn then
          mmoDriver.Lines.Add('Wake up is on')
        else
          mmoDriver.Lines.Add('Wake up is off');
        Data := (Data shr CFlexCopRevisionShift) and CFlexCopRevisionAnd;
        case Data of
          $40: mmoDriver.Lines.Add('FlexCop revision 064');
          $82: mmoDriver.Lines.Add('FlexCop revision 130 (FLEXCOP II)');
          $C0: mmoDriver.Lines.Add('FlexCop revision 192 (FLEXCOP III)');
          $C2: mmoDriver.Lines.Add('FlexCop revision 194');
          $C3: mmoDriver.Lines.Add('FlexCop revision 195 (FLEXCOP IIB)');
          else mmoDriver.Lines.Add(format('Unknown FlexCop revision %3.3d', [Data]));
        end;
      end
      else
        mmoDriver.Lines.Add('Error accessing driver');

      // Get B2C2 identication
      SetLength(ByteArray, CFlexCopI2cEepromB2c2AddressBytes);
      if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromB2c2Address, ByteArray) then
      begin
        Identification := ByteArray[0];
        Identification := (Identification shl 8) or ByteArray[1];
        mmoDriver.Lines.Add(format('EEPROM: Identification: %4.4x', [Identification]))
      end
      else
        mmoDriver.Lines.Add('Error reading MAC address from EEPROM');

      // RAM
      Size := FlexCopRamDetectSize(CardHandle);
      if Size = $FFFF then
        mmoDriver.Lines.Add('RAM size could not be determined')
      else
      begin
        Error := False;
        mmoDriver.Lines.Add(format('RAM size: %d kB', [Size]));
        for Loop := 0 to (1024 * Size)-1 do
        begin
          AByte := (Loop and $FF) xor ((Loop shr 8) and $FF) xor  ((Loop shr 16) and $FF);
          if not FlexCopRamWrite(CardHandle, Loop, AByte) then
          begin
            mmoDriver.Lines.Add(format('RAM writing interupted at: $%x.', [Loop]));
            Error := True;
            Break;
          end;
        end;
        for Loop := 0 to (1024 * Size)-1 do
        begin
          if not FlexCopRamRead(CardHandle, Loop, AByte) then
          begin
            mmoDriver.Lines.Add(format('RAM reading interupted at: $%x.', [Loop]));
            Break;
          end;
          CheckByte := ((Loop and $FF) xor ((Loop shr 8) and $FF) xor ((Loop shr 16) and $FF));
          if CheckByte <> AByte then
          begin
            mmoDriver.Lines.Add(format('RAM check error at address $%5.5x (read $%2.2x but was expecting $%2.2x).', [Loop, AByte, CheckByte]));
            Error := True;
            Break;
          end;
        end;
        if not Error then
          mmoDriver.Lines.Add('RAM check done. No errors found.');
      end;

      // Get MAC address
      SetLength(ByteArray, CFlexCopI2cEepromMacAddressBytes);
      if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromMacAddress, ByteArray) then
      begin
        MacAddress := ByteArray[0];
        MacAddress := (MacAddress shl 8) or ByteArray[1];
        MacAddress := (MacAddress shl 8) or ByteArray[2];
        MacAddress := (MacAddress shl 8) or ByteArray[3];
        MacAddress := (MacAddress shl 8) or ByteArray[4];
        MacAddress := (MacAddress shl 8) or ByteArray[5];
        mmoDriver.Lines.Add(format('EEPROM: Mac address: $%12.12x, checksum $%2.2x', [MacAddress, FlexCopEepromChecksum(ByteArray)]))
      end
      else
        mmoDriver.Lines.Add('Error reading MAC address from EEPROM');

      // Get tuner type (if any)
      SetLength(ByteArray, CFlexCopI2cEepromTunerAddressBytes);
      if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromTunerAddress, ByteArray) then
      begin
        TunerType := ByteArray[0];
        TunerType := (TunerType shl 8) or ByteArray[1];
        TunerType := (TunerType shl 8) or ByteArray[2];
        TunerType := (TunerType shl 8) or ByteArray[3];
        mmoDriver.Lines.Add(format('EEPROM: Tuner type: $%8.8x, checksum $%2.2x', [TunerType, FlexCopEepromChecksum(ByteArray)]));
      end
      else
        mmoDriver.Lines.Add('Error reading tuner type from EEPROM');

      // Get keys
      SetLength(ByteArray, CFlexCopI2cEepromKeyAddressBytes);
      if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromTunerAddress, ByteArray) then
      begin
        NewStr := '';
        for Loop := 0 to CFlexCopI2cEepromKeyAddressBytes-2 do
          NewStr := NewStr + format('%2.2x', [ByteArray[Loop]]);
        mmoDriver.Lines.Add(format('EEPROM: keys: $%s, checksum $%2.2x', [NewStr, FlexCopEepromChecksum(ByteArray)]));
      end
      else
        mmoDriver.Lines.Add('Error reading keys from EEPROM');

      if not FlexCopWriteEeprom(CardHandle, $300, ByteArray) then
        mmoDriver.Lines.Add('Error writing data to EEPROM');

      if FlexCopCheckQpskResetValues(CardHandle) then
        mmoDriver.Lines.Add('QPSK values comply with reset values')
      else
        mmoDriver.Lines.Add('QPSK values do not comply with reset values (tuner might not have been switched off long enough for a reset)');

      if not FlexCopWriteDefaultsToQpsk(CardHandle, 0) then
        mmoDriver.Lines.Add('Error writing default values to QPSK')
      else
        mmoDriver.Lines.Add('Default values set for QPSK');

      if FlexCopCheckQpskDefaultValues(CardHandle, 0) then
        mmoDriver.Lines.Add('QPSK values comply with default values')
      else
        mmoDriver.Lines.Add('QPSK values do not comply with default values (migth be normal)');

      SynthesizerAddress := FlexCopDetectSynthesizerAddress(CardHandle);
      if SynthesizerAddress = 255 then
        mmoDriver.Lines.Add('Could not detect the synthesizer')
      else
      begin
        mmoDriver.Lines.Add(format('Synthesizer detected at index %d', [SynthesizerAddress]));
        if not FlexCopReadFromSynthesizer(CardHandle, SynthesizerAddress, AByte) then
          mmoDriver.Lines.Add('Could not read from synthesizer')
        else
          mmoDriver.Lines.Add(format('SL1935 synthesizer status: $%2.2x', [AByte]));
      end;

    finally
      FlexCopCloseHandle(CardHandle);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  //
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


end.
