{******************************************************************************}
{ FileName............: Dvb                                                    }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  DVB related                                                                 }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Notes: Record typically contain ALL information, including 'overlayed'       }
{        information                                                           }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030823 - Initial release                                           }
{******************************************************************************}

unit Dvb;

interface

uses
  Windows;

const
  CDvbPacketSize = 204 - 16;
    // DVB packet size: parity is stripped by the SAA7146A
  CDvbPacketSyncWord = $47; // DVB sync word
  CDvbPacketHSync = CDvbPacketSize;
    // 'Horizontal' synchronization DD1 of SAA7146A
  CDvbPacketVSync = 512; // 'Vertical'   synchronization DD1 of SAA7146A
  CDvbPacketBufferSize = CDvbPacketHSync * CDvbPacketVSync;
    // All packets in a single buffer

  CDvbTransportPidPAT = $0000; // Program Association Table identifier
  CDvbTransportPidCAT = $0001; // Conditional Access Table identifier
  CDvbTransportPidPMT = $0002; // Program Map Table identifier
  CDvbTransportPidAssignStart = $0010; // First Pid user assignable
  CDvbTransportPidAssignEnd = $1FFE; // Last  Pid user assignable
  CDvbTransportPidNull = $1FFF; // Null packet

  CDvbTypePidUndefined = $00; // Undefined
  CDvbTypePidPmt = $02; // Program Map Table PID
  CDvbTypePidPcr = $03; // Program Clock Reference PID
  CDvbTypePidPes = $FF; // PES Stream

  // StreamId assignments
  CDvbProgramStreamMap = $BC;
  CDvbPrivateStream1 = $BD;
  CDvbPaddingStream = $BE;
  CDvbPrivateStream2 = $BF;
  CDvbEcmStream = $F0;
  CDvbEmmStream = $F1;
  CDvbDsmccStream = $F2;
  CDvbTypeEStream = $F8;
  CDvbProgramStreamDirectory = $FF;

  // TrickModeControl
  CDvbTrickModeFastForward = $00;
  CDvbTrickModeSlowMotion = $01;
  CDvbTrickModeFreezeFrame = $02;
  CDvbTrickModeFastReverse = $03;
  CDvbTrickModeSlowReverse = $04;

type
  PDvbTransportPacket = ^TDvbTransportPacket;
  TDvbTransportPacket = array[0..CDvbPacketSize - 1] of Byte;

  PDvbTransportPackets = ^TDvbTransportPackets;
  TDvbTransportPackets = array[0..CDvbPacketVSync - 1] of TDvbTransportPacket;

  TDvbTransportPids = array[0..$1FFF] of Boolean;
  TDvbMaxSection = array[0..1023] of Byte;
  TDvbMaxPesPacket = array[0..$FFFF] of Byte;

  TDvbTypePid = record
    PidType: Byte; // Type of Pid
    // $00 = undefined (PAT reserved)
    // $01 = undefined (CAT reserved)
    // $02 = PMT
    // $03 = PCR
    // $FF = PES
    ProgramNumber: Word; // Associated program number
    StreamType: Byte; // Associated stream type
    EsInfoLength: Word; // Associated length of descriptor
    Descriptor: TDvbMaxSection; // Associated descriptor
  end;
  TDvbTypeOfPids = array[0..$1FFF] of TDvbTypePid;

  // Transport stream header individual components
  TDvbTransportStreamHeader = record
    HeaderAndPayload: TDvbTransportPacket;
    TransportValid: Boolean;
      // Indicates no obvious errors found, eg. SyncByte/Error indicator
    Sync: Boolean; // Indiactes we have seen a starting packet

    SyncByte: Byte; // Should be $47
    TransportErrorIndicator: Boolean;
    PayloadUnitStartIndicator: Boolean;
    TransportPriority: Boolean;
    Pid: Word;
    TransportScramblingControl: Byte;
    AdaptationFieldControl: Byte;
    ContinuityCounter: Byte;
    AdaptationFieldLength: Byte;
    DiscontinuityIndicator: Boolean;
    RandomAccessIndicator: Boolean;
    ElementaryStreamPriorityIndicator: Boolean;
    PcrFlag: Boolean;
    OpcrFlag: Boolean;
    SplicingPointFlag: Boolean;
    TransportPrivateDataFlag: Boolean;
    AdaptationFieldExtensionFlag: Boolean;
    ProgramClockReferenceBase: Int64;
    PcrReserved: Byte;
    ProgramClockReferenceExtension: Word;
    OriginalProgramClockReferenceBase: Int64;
    OpcrReserved: Byte;
    OriginalProgramClockReferenceExtension: Word;
    SpliceCountdown: Byte;
    TransportPrivateDataLength: Byte;
    PrivateDataByte: array[0..255] of Byte;
    AdaptationFieldExtensionLength: Byte;
    LtwFlag: Boolean;
    PiecewiseRateFlag: Boolean;
    SeamlessSpliceFlag: Boolean;
    AdaptationExtensionReserved: Byte;
    LtwValidFlag: Boolean;
    LtwOffset: Word;
    PiecewiseReserved: Byte;
    PiecewiseRate: Dword;
    SpliceType: Byte;
    DtsNextAu: Int64;
    DataBytes: array[0..183] of Byte;
    DataBytesLength: Word;
  end;

  // Raw received data (should be somewhat larger?)
  TDvbPesPacketRaw = TDvbMaxPesPacket;
  PTDvbProgramElementaryStream = ^TDvbProgramElementaryStream;
  // PES packet individual components
  TDvbProgramElementaryStream = record
    RawData: TDvbPesPacketRaw; // Raw, unprocessed, data of multiple packets
    RawIndex: Word; // Index in raw data (save)
    TableValid: Boolean; // Indicates valid data (eg. checksum correct)
    Sync: Boolean; // Indiactes we have seen a starting packet

    PacketStartCodePrefix: Dword;
    StreamId: Byte;
    PesPacketLength: Word;
    Pes10: Byte;
    PesScramblingControl: Byte;
    PesPriority: Boolean;
    DataAlignmentIndicator: Boolean;
    Copyright: Boolean;
    OriginalOrCopy: Boolean;
    PtsDtsFlags: Byte;
    EscrFlag: Boolean;
    EsRateFlag: Boolean;
    DsmTrickModeFlag: Boolean;
    AdditionalCopyInfoFlag: Boolean;
    PesCrcFlag: Boolean;
    PesExtensionFlag: Boolean;
    PesHeaderDataLength: Byte;

    Pes0010: Byte;
    Pts00103230: Byte;
    MarkerBit00103230: Boolean;
    Pts00102915: Word;
    MarkerBit00102915: Boolean;
    Pts00101400: Word;
    MarkerBit00101400: Boolean;

    Pes0011: Byte;
    Pts00113230: Byte;
    MarkerBit00113230: Boolean;
    Pts00112915: Word;
    MarkerBit00112915: Boolean;
    Pts00111400: Word;
    MarkerBit00111400: Boolean;

    Pes0001: Byte;
    Pts00013230: Byte;
    MarkerBit00013230: Boolean;
    Pts00012915: Word;
    MarkerBit00012915: Boolean;
    Pts00011400: Word;
    MarkerBit00011400: Boolean;

    Reserved: Byte;
    EscrBase3230: Byte;
    MarkerBit3230: Boolean;
    EscrBase2915: Word;
    MarkerBit2915: Boolean;
    EscrBase1400: Word;
    MarkerBit1400: Boolean;
    EscrExtension: Word;
    MarkerBitExtension: Boolean;

    MarkerBitEsRateFlag: Boolean;
    EsRate: Dword;
    MarkerBitEsRate: Boolean;

    TrickModeControl: Byte;
    FieldIdFf: Byte;
    IntraSliceRefreshFf: Boolean;
    FrequencyTruncationFf: Byte;
    RepCntrlSm: Byte;
    FieldIdFrf: Byte;
    ReservedFrf: Byte;
    FieldIdFr: Byte;
    IntraSliceRefreshFr: Boolean;
    FrequencyTruncationFr: Byte;
    RepCntrlSr: Byte;
    ReservedTrickMode: Byte;

    MarkerBitAdditionalCopyFlag: Boolean;
    AdditionalCopyInfo: Byte;

    PreviousPesPacketCrc: Word;

    PesPrivateDataFlag: Boolean;
    PackHeaderFieldFlag: Boolean;
    ProgramPacketSequenceCounterFlag: Boolean;
    PstdBufferFlag: Boolean;
    ReservedPesExtension: Byte;
    PesExtensionFlag2: Boolean;
    PesPrivateData: array[0..15] of Byte;
    PackFieldLength: Byte;
    PackHeader: array[0..255] of Byte;

    MarkerBitPpsc: Boolean;
    ProgramPacketSequenceCounter: Byte;
    MarkerBitPpsc2: Boolean;
    Mpeg1Mpeg2Identifier: Boolean;
    OriginalStuffLength: Byte;

    Pstd01: Byte;
    PstdBufferScale: Boolean;
    PstdBufferSize: Word;

    MarkerBitPef: Boolean;
    PesExtensionFieldLength: Byte;
    ReservedPesExtensionField: array[0..127] of Byte;

    StuffingByte: array[0..63] of Byte;
    PesPacketDataByte: TDvbMaxPesPacket;
  end;

  // Raw received data
  TDvbProgramAssociationRaw = TDvbMaxSection;

  // Program Association Table PID part
  TDvbProgramAssociationPid = record
    ProgramNumber: Word;
    Reserved: Byte;
    NetworkPid: Word;
    ProgramMapPid: Word;
  end;

  // Program Association Table
  TDvbProgramAssociationTable = record
    RawData: TDvbProgramAssociationRaw;
      // Raw, unprocessed, data of multiple packets
    RawIndex: Word; // Index in raw data (save)
    ProgramIndex: Byte; // Index in <ProgramData> array
    TableValid: Boolean; // Indicates valid data (eg. checksum correct)
    Sync: Boolean; // Indiactes we have seen a starting packet

    TableId: Byte;
    SectionSyntaxIndicator: Boolean;
    Zero: Boolean;
    Reserved: Byte;
    SectionLength: Word;
    TransportStreamId: Word;
    Reserved2: Byte;
    VersionNumber: Byte;
    CurrentNextIndicator: Boolean;
    SectionNumber: Byte;
    LastSectionNumber: Byte;
    ProgramData: array[0..255] of TDvbProgramAssociationPid;
    Crc32: Dword;
  end;

  // Raw received data
  TDvbConditionalAccessRaw = TDvbMaxSection;

  // Conditional Access
  TDvbConditionalAccessTable = record
    RawData: TDvbProgramAssociationRaw;
      // Raw, unprocessed, data of multiple packets
    RawIndex: Word; // Index in raw data (save)
    TableValid: Boolean; // Indicates valid data (eg. checksum correct)
    Sync: Boolean; // Indiactes we have seen a starting packet

    TableId: Byte;
    SectionSyntaxIndicator: Boolean;
    Zero: Boolean;
    Reserved: Byte;
    SectionLength: Word;
    Reserved3: Word;
    Reserved2: Byte;
    VersionNumber: Byte;
    CurrentNextIndicator: Boolean;
    SectionNumber: Byte;
    LastSectionNumber: Byte;
    DescriptorData: TDvbMaxSection;
    Crc32: Dword;
  end;

  // Raw received data
  TDvbProgramMapRaw = TDvbMaxSection;

  // Program map streams
  TDvbProgramMapStream = record
    DescriptorIndex: Word;

    StreamType: Byte;
    Reserved: Byte;
    ElementaryPid: Word;
    Reserved2: Byte;
    EsInfoLength: Word;
    Descriptor: TDvbMaxSection;
  end;

  // Program Map Table
  PTDvbProgramMapTable = ^TDvbProgramMapTable;
  TDvbProgramMapTable = record
    RawData: TDvbProgramAssociationRaw;
      // Raw, unprocessed, data of multiple packets
    RawIndex: Word; // Index in raw data (save)
    StreamIndex: Word; // Index in <StreamData> array
    TableValid: Boolean; // Indicates valid data (eg. checksum correct)
    Sync: Boolean; // Indiactes we have seen a starting packet

    TableId: Byte;
    SectionSyntaxIndicator: Boolean;
    Zero: Boolean;
    Reserved: Byte;
    SectionLength: Word;
    ProgramNumber: Word;
    Reserved2: Byte;
    VersionNumber: Byte;
    CurrentNextIndicator: Boolean;
    SectionNumber: Byte;
    LastSectionNumber: Byte;
    Reserved3: Byte;
    PcrPid: Word;
    Reserved4: Byte;
    ProgramInfoLength: Word;
    DescriptorData: TDvbMaxSection;
    StreamData: array[0..255] of TDvbProgramMapStream;
    Crc32: Dword;
  end;

  // Program Map Table
  TDvbProgramMapTables = array[0..$FFFF] of PTDvbProgramMapTable;
  // Program Map Table
  TDvbProgramElementaryStreams = array[0..$FFFF] of
    PTDvbProgramElementaryStream;

  // Program number list
  TDvbProgramNumbers = array[0..$FFFF] of Word;

function DvbResetValueCrc16: Word;
function DvbResetValueCrc32: Dword;
function DvbNewValueCrc16(Checksum: Word; Data: Byte): Word;
function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword;

implementation

var
  Crc16Table: array[0..255] of Word;
  Crc32Table: array[0..255] of Dword;

  // Next table is for reference only. This is the generated data and should
  // be equal to <Crc32Table>.
  Crc32Table2: array[0..255] of Dword =
  ($00000000, $04C11DB7, $09823B6E, $0D4326D9, $130476DC, $17C56B6B,
    $1A864DB2, $1E475005, $2608EDB8, $22C9F00F, $2F8AD6D6, $2B4BCB61,
    $350C9B64, $31CD86D3, $3C8EA00A, $384FBDBD, $4C11DB70, $48D0C6C7,
    $4593E01E, $4152FDA9, $5F15ADAC, $5BD4B01B, $569796C2, $52568B75,
    $6A1936C8, $6ED82B7F, $639B0DA6, $675A1011, $791D4014, $7DDC5DA3,
    $709F7B7A, $745E66CD, $9823B6E0, $9CE2AB57, $91A18D8E, $95609039,
    $8B27C03C, $8FE6DD8B, $82A5FB52, $8664E6E5, $BE2B5B58, $BAEA46EF,
    $B7A96036, $B3687D81, $AD2F2D84, $A9EE3033, $A4AD16EA, $A06C0B5D,
    $D4326D90, $D0F37027, $DDB056FE, $D9714B49, $C7361B4C, $C3F706FB,
    $CEB42022, $CA753D95, $F23A8028, $F6FB9D9F, $FBB8BB46, $FF79A6F1,
    $E13EF6F4, $E5FFEB43, $E8BCCD9A, $EC7DD02D, $34867077, $30476DC0,
    $3D044B19, $39C556AE, $278206AB, $23431B1C, $2E003DC5, $2AC12072,
    $128E9DCF, $164F8078, $1B0CA6A1, $1FCDBB16, $018AEB13, $054BF6A4,
    $0808D07D, $0CC9CDCA, $7897AB07, $7C56B6B0, $71159069, $75D48DDE,
    $6B93DDDB, $6F52C06C, $6211E6B5, $66D0FB02, $5E9F46BF, $5A5E5B08,
    $571D7DD1, $53DC6066, $4D9B3063, $495A2DD4, $44190B0D, $40D816BA,
    $ACA5C697, $A864DB20, $A527FDF9, $A1E6E04E, $BFA1B04B, $BB60ADFC,
    $B6238B25, $B2E29692, $8AAD2B2F, $8E6C3698, $832F1041, $87EE0DF6,
    $99A95DF3, $9D684044, $902B669D, $94EA7B2A, $E0B41DE7, $E4750050,
    $E9362689, $EDF73B3E, $F3B06B3B, $F771768C, $FA325055, $FEF34DE2,
    $C6BCF05F, $C27DEDE8, $CF3ECB31, $CBFFD686, $D5B88683, $D1799B34,
    $DC3ABDED, $D8FBA05A, $690CE0EE, $6DCDFD59, $608EDB80, $644FC637,
    $7A089632, $7EC98B85, $738AAD5C, $774BB0EB, $4F040D56, $4BC510E1,
    $46863638, $42472B8F, $5C007B8A, $58C1663D, $558240E4, $51435D53,
    $251D3B9E, $21DC2629, $2C9F00F0, $285E1D47, $36194D42, $32D850F5,
    $3F9B762C, $3B5A6B9B, $0315D626, $07D4CB91, $0A97ED48, $0E56F0FF,
    $1011A0FA, $14D0BD4D, $19939B94, $1D528623, $F12F560E, $F5EE4BB9,
    $F8AD6D60, $FC6C70D7, $E22B20D2, $E6EA3D65, $EBA91BBC, $EF68060B,
    $D727BBB6, $D3E6A601, $DEA580D8, $DA649D6F, $C423CD6A, $C0E2D0DD,
    $CDA1F604, $C960EBB3, $BD3E8D7E, $B9FF90C9, $B4BCB610, $B07DABA7,
    $AE3AFBA2, $AAFBE615, $A7B8C0CC, $A379DD7B, $9B3660C6, $9FF77D71,
    $92B45BA8, $9675461F, $8832161A, $8CF30BAD, $81B02D74, $857130C3,
    $5D8A9099, $594B8D2E, $5408ABF7, $50C9B640, $4E8EE645, $4A4FFBF2,
    $470CDD2B, $43CDC09C, $7B827D21, $7F436096, $7200464F, $76C15BF8,
    $68860BFD, $6C47164A, $61043093, $65C52D24, $119B4BE9, $155A565E,
    $18197087, $1CD86D30, $029F3D35, $065E2082, $0B1D065B, $0FDC1BEC,
    $3793A651, $3352BBE6, $3E119D3F, $3AD08088, $2497D08D, $2056CD3A,
    $2D15EBE3, $29D4F654, $C5A92679, $C1683BCE, $CC2B1D17, $C8EA00A0,
    $D6AD50A5, $D26C4D12, $DF2F6BCB, $DBEE767C, $E3A1CBC1, $E760D676,
    $EA23F0AF, $EEE2ED18, $F0A5BD1D, $F464A0AA, $F9278673, $FDE69BC4,
    $89B8FD09, $8D79E0BE, $803AC667, $84FBDBD0, $9ABC8BD5, $9E7D9662,
    $933EB0BB, $97FFAD0C, $AFB010B1, $AB710D06, $A6322BDF, $A2F33668,
    $BCB4666D, $B8757BDA, $B5365D03, $B1F740B4);

const
  // 16 Bit CCITT standard polynomial x^16 + x^12 + x^5 + x^1 + x^0
  CDvbPolynomial16 = $1021;
  CDvbCrcInitial16 = $FFFF;
  // 32 Bit standard polynomial x^32 + x^26 + x^23 + x^22 + x^16 +
  //   x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x^1 + x^0
  CDvbPolynomial32 = $04C11DB7;
  CDvbCrcInitial32 = $FFFFFFFF;

  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: Generate CRC tables
    Notes   :
  ------------------------------------------------------------------------------}

procedure DvbGenerateCrcTables;
var
  ByteLoop: Byte;
  BitLoop: Byte;
  Crc16: Word;
  Crc32: Dword;
begin
  // Go through the whole array
  for ByteLoop := 0 to 255 do
  begin
    Crc16 := ByteLoop shl 8;
    Crc32 := ByteLoop shl 24;
    for BitLoop := 0 to 7 do
    begin
      if (Crc16 and $8000) <> 0 then
        Crc16 := ((Crc16 and $7FFF) shl 1) xor CDvbPolynomial16
      else
        Crc16 := (Crc16 shl 1);
      if (Crc32 and $80000000) <> 0 then
        Crc32 := ((Crc32 and $7FFFFFFF) shl 1) xor CDvbPolynomial32
      else
        Crc32 := (Crc32 shl 1);
      Crc16Table[ByteLoop] := Crc16;
      Crc32Table[ByteLoop] := Crc32;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc16: Word;
begin
  Result := CDvbCrcInitial16;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc32: Dword;
begin
  Result := CDvbCrcInitial32;
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC16 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial16> as
            starting value
------------------------------------------------------------------------------}

function DvbNewvalueCrc16(Checksum: Word; Data: Byte): Word;
begin
  Result := ((Checksum and $FF) shl 8) xor Crc16Table[((Checksum shr 8) xor Data)
    and $FF];
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC32 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial32> as
            starting value
------------------------------------------------------------------------------}

function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword;
begin
  Result := ((Checksum and $FFFFFF) shl 8) xor Crc32Table[((Checksum shr 24) xor
    Data) and $FF];
end;

initialization
  // The CRC tables are generated at startup
  DvbGenerateCrcTables;

finalization

end.

