{******************************************************************************}
{ FileName............: UFlexCopApiFrontEnd                                    }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Front end card interface (based on Linux API V3).                           }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{  Frontend function calls                                                     }
{  . Open()                                                                    }
{  . Close()                                                                   }
{  . IoCtl()                                                                   }
{    - FE_READ_STATUS                                                          }
{    - FE_READ_BER                                                             }
{    - FE_READ_SNR                                                             }
{    - FE_READ_SIGNAL_STRENGTH                                                 }
{    - FE_READ_UNCORRECTED_BLOCKS                                              }
{    - FE_SET_FRONTEND                                                         }
{    - FE_GET_FRONTEND                                                         }
{    - FE_GET_EVENT                                                            }
{    - FE_GET_INFO                                                             }
{    - FE_DISEQC_RESET_OVERLOAD                                                }
{    - FE_DISEQC_SEND_MASTER_CMD                                               }
{    - FE_DISEQC_RECV_SLAVE_REPLY                                              }
{    - FE_DISEQC_SEND_BURST                                                    }
{    - FE_SET_TONE                                                             }
{    - FE_SET_VOLTAGE                                                          }
{    - FE_ENABLE_HIGH_LNB_VOLTAGE                                              }
{    - FE_DISEQC_SEND_BURST2                                                   }
{    - FE_SET_TONE2                                                            }
{    - FE_SET_VOLTAGE2                                                         }
{    - FE_ENABLE_HIGH_LNB_VOLTAGE2                                             }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   200504167 - Initial release                                           }
{******************************************************************************}
unit UFlexCopFrontEnd;
interface

  function Open (const DeviceName: PChar; const Flags: Integer)                            : Integer; stdcall;
  function Close(const FileDescriptor: Integer)                                            : Integer; stdcall;
  function IoCtl(const FileDescriptor: Integer; const Request: Integer; Parameter: Pointer): Integer; stdcall;

implementation

uses
  FlexCopFrontEndDefines,
  FlexCopI2c, FlexCopInterface,
  FlexCopRegisters, FlexCopGeneral,
  Stv0299bRegisters, Sl1935Registers,
{$IFDEF USELOG}
  Classes,
{$ENDIF}
  IniFiles, SysUtils,  Windows;


type
  IoCtlCalls = (IoCtl_FE_GET_INFO,                IoCtl_FE_DISEQC_RESET_OVERLOAD,
                IoCtl_FE_DISEQC_SEND_MASTER_CMD , IoCtl_FE_DISEQC_RECV_SLAVE_REPLY,
                IoCtl_FE_DISEQC_SEND_BURST      , IoCtl_FE_SET_TONE,
                IoCtl_FE_SET_VOLTAGE            , IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE,
                IoCtl_FE_READ_STATUS            , IoCtl_FE_READ_BER,
                IoCtl_FE_READ_SIGNAL_STRENGTH   , IoCtl_FE_READ_SNR,
                IoCtl_FE_READ_UNCORRECTED_BLOCKS, IoCtl_FE_SET_FRONTEND,
                IoCtl_FE_GET_FRONTEND           , IoCtl_FE_GET_EVENT,
                // Additional functions which use different paramaters than the original
                IoCtl_FE_DISEQC_SEND_BURST2     , IoCtl_FE_SET_TONE2,
                IoCtl_FE_SET_VOLTAGE2           , IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE2
                );

var
  IoCtlIdentifier      : array[Ord(IoCtl_FE_GET_INFO)..Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE2)] of Integer;// IoCtl's to act upon

  TunerType            : Byte;                   // Tuner type

  IniFile              : TMemIniFile;            // Ini file

{$IFDEF USELOG}
  LogStream            : TFileStream;            // Filestream for log
  LogLevel             : Byte;                   // LogLevel (00 = no log)
                                                 //  $x0 = No log
                                                 //  $x1 = Lowest level
                                                 //  $x2 = Command errors
                                                 //  $x3 = Procedure/Function entries
                                                 //  $x4 = Procedure/Function specifics
                                                 //  $x5 = Generic procedure/Function entries


{------------------------------------------------------------------------------
  Params  : <LogString>  Data to log
            <Level>      Log level: if this number if <= <LogLevel> then
                         it is logged
                         $8x indicates that the string is intended for
                             displaying on form too.
  Returns : -

  Descript: Write data to log. A timestamp is added.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ToLog(LogString: string; Level: Byte);
var
  NewLog: string;
begin
  try
    // Only log if the level is high enough
    if (Level and $0F) > LogLevel then
      Exit;
    // The check on <LogStream> is necessary because the application might
    // call this prior to completing initialization
    if not Assigned(LogStream) then
      Exit;
    NewLog := FormatDateTime('YYYYMMDD"T"HHMMSS"  "', Now) + LogString + #13#10;
    LogStream.Write(NewLog[1], Length(NewLog));
  except
  end;  
end;
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : <Section>    Section to access in INI file
            <Paremeter>  Paremeter in INI file
            <Default>    Default value (eg. if not present in INI file)
  Returns : <Result>     String value for parameter

  Descript: Get parameter from INI file.
  Notes   :
 ------------------------------------------------------------------------------}
function GetParameter(Section: string; Parameter: string; Default: string): string;
var
  FromIni : string;
  Position: Integer;
  TempStr : string;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd GetParameter (Section: [' + Section + '], Parameter: [' + Parameter + '].', $03);
{$ENDIF}
    if not Assigned(IniFile) then
      Exit;
    FromIni := IniFile.ReadString(Section, Parameter, Default);
    // Strip out leading/trailing spaces
    FromIni := Trim(FromIni);
    // Check for starting string parameter
    Position := Pos('''', FromIni);
    if Position = 1 then
    begin
      TempStr  := Copy(FromIni, Position+1, 255);
      // Find ending of string
      Position := Pos('''', TempStr);
      if Position <> 0 then
      begin
        // End of string found, which will be the result
        Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
        ToLog('FlexCopFrontEnd Result: [' + Result + '].', $05);
{$ENDIF}
        Exit;
      end;
    end
    else
    begin
      Position := Pos('"', FromIni);
      if Position = 1 then
      begin
        TempStr := Copy(FromIni, Position+1, 255);
        // Find ending of string
        Position := Pos('"', TempStr);
        if Position <> 0 then
        begin
          // End of string found, which will be the result
          Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
          ToLog('FlexCopFrontEnd Result: [' + Result + '].', $05);
{$ENDIF}
          Exit;
        end;
      end;
    end;
    // We know we don't have a string type so handle the whole string as
    // normal text. We could have a comment in it so check this too
    Position := Pos(';', FromIni);
    if Position <> 0 then
    begin
      TempStr := Copy(FromIni, 1, Position-1);
      // Strip out leading/trailing spaces
      Result := Trim(TempStr);
    end
    else
      Result := FromIni;
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd Result: [' + Result + '].', $05);
{$ENDIF}
  except
  end;
end;


{******************************************************************************}
{                              LINUX API FUNCTIONS                             }
{                                                                              }
{ Note: Minimal checks are done here because most of the checks are already    }
{       handled by the caller function <IoCtl>.                                }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to front end info structure
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end get info
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeGetInfo(FileDescriptor: Integer; Request: Integer; Parameter: PDvbFrontEndInfo): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_GET_INFO.', $04);
{$ENDIF}
    CopyMemory(Parameter, @Stv0299bDvbFrontEndInfo, SizeOf(TDvbFrontEndInfo));
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end DiSEqC reset overload
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeDiSEqCResetOverload(FileDescriptor: Integer; Request: Integer): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_DISEQC_RESET_OVERLOAD.', $04);
    ToLog('FlexCopFrontEnd Not supported.', $05);
{$ENDIF}
    Result := -EOPNOTSUPP;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to master command structure
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end DiSEqC send master command
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeDiSEqCSendMasterCmd(FileDescriptor: Integer; Request: Integer; Parameter: PDvbDiSEqCMasterCmd): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_DISEQC_SEND_MASTER_CMD.', $04);
{$ENDIF}
    // Check size
    if Parameter.MsgLen > SizeOf(TDiSEqcData) then
    begin
      Result := -EINVAL;
      Exit;
    end;
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd DiSEqC length %d, Data %d %d %d %d %d %d.', [Parameter.MsgLen, Parameter.Msg[0],
      Parameter.Msg[1], Parameter.Msg[2], Parameter.Msg[3], Parameter.Msg[4], Parameter.Msg[5]]), $04);
{$ENDIF}
    if FlexCopDiSEqCCommand2(FileDescriptor, CDiSEqCCommand, Parameter.MsgLen, TDiSEqCData(Parameter.Msg)) then
      Result := ENOERROR
    else
      Result := -EPERM;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to slave reply structure
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end DiSEqC receive slave reply
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeDiSEqCRecvSlaveReply(FileDescriptor: Integer; Request: Integer; Parameter: PDvbDiSEqCSlaveReply): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_DISEQC_RECV_SLAVE_REPLY.', $04);
    ToLog('FlexCopFrontEnd Not supported.', $04);
{$ENDIF}
    Result := -EOPNOTSUPP;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Burst
                              . A (SEC_MINI_A)
                              . B (SEC_MINI_B)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end DiSEqC send burst
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeDiSEqCSendBurst(FileDescriptor: Integer; Request: Integer; Parameter: Integer): Integer;
var
  DiSEqcData: TDiSEqCData;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_DISEQC_SEND_BURST.', $04);
{$ENDIF}
    case Parameter of
      SEC_MINI_A: begin
{$IFDEF USELOG}
                    ToLog('FlexCopFrontEnd SEC_MINI_A.', $05);
{$ENDIF}
                    if FlexCopDiSEqCCommand2(FileDescriptor, CDiSEqCBurstA, 0, DiSEqCData) then
                      Result := ENOERROR
                    else
                      Result := -EPERM;
                  end;
      SEC_MINI_B: begin
{$IFDEF USELOG}
                    ToLog('FlexCopFrontEnd SEC_MINI_B.', $05);
{$ENDIF}
                    if FlexCopDiSEqCCommand2(FileDescriptor, CDiSEqCBurstB, 0, DiSEqCData) then
                      Result := ENOERROR
                    else
                      Result := -EPERM;
                  end;
      else        begin
{$IFDEF USELOG}
                    ToLog('FlexCopFrontEnd Unknown burst type.', $04);
{$ENDIF}
                    Result := -EINVAL;
                  end;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to Burst
                              . A (SEC_MINI_A)
                              . B (SEC_MINI_B)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end DiSEqC send burst
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeDiSEqCSendBurst2(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
begin
  try
    Result := IoCtlFeDiSEqCSendBurst(FileDescriptor, Request, Parameter^);
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Tone generation mode
                              . on  (SEC_TONE_ON)
                              . off (SEC_TONE_OFF)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end set tone
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeSetTone(FileDescriptor: Integer; Request: Integer; Parameter: Integer): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_SET_TONE.', $04);
{$ENDIF}
    case Parameter of
      SEC_TONE_ON:  begin
{$IFDEF USELOG}
                      ToLog('FlexCopFrontEnd SEC_TONE_ON.', $05);
{$ENDIF}
                      if FlexCopHighBandLNB(FileDescriptor) then
                        Result := ENOERROR
                      else
                       Result := -EPERM;
                    end;
      SEC_TONE_OFF: begin
{$IFDEF USELOG}
                      ToLog('FlexCopFrontEnd SEC_TONE_OFF.', $05);
{$ENDIF}
                      if FlexCopLowBandLNB(FileDescriptor) then
                        Result := ENOERROR
                      else
                        Result := -EPERM;
                    end;
      else          begin
{$IFDEF USELOG}
                      ToLog('FlexCopFrontEnd Unknown tone setting.', $05);
{$ENDIF}
                      Result := -EINVAL;
                    end;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to tone generation mode
                              . on  (SEC_TONE_ON)
                              . off (SEC_TONE_OFF)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end set tone
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeSetTone2(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
begin
  try
    Result := IoCtlFeSetTone(FileDescriptor, Request, Parameter^);
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Bus voltage
                              . 13V (SEC_VOLTAGE_13)
                              . 18V (SEC_VOLTAGE_18)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end set voltage
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeSetVoltage(FileDescriptor: Integer; Request: Integer; Parameter: Integer): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_SET_VOLTAGE.', $04);
{$ENDIF}
    case Parameter of
      SEC_VOLTAGE_13: begin
{$IFDEF USELOG}
                        ToLog('FlexCopFrontEnd SEC_VOLTAGE_13.', $05);
{$ENDIF}
                        if FlexCopVerticalPolarityLNB(FileDescriptor) then
                          Result := ENOERROR
                        else
                          Result := -EPERM;
                      end;
      SEC_VOLTAGE_18: begin
{$IFDEF USELOG}
                        ToLog('FlexCopFrontEnd SEC_VOLTAGE_18.', $05);
{$ENDIF}
                        if FlexCopHorizontalPolarityLNB(FileDescriptor) then
                          Result := ENOERROR
                        else
                          Result := -EPERM;
                      end;
      else            begin
{$IFDEF USELOG}
                        ToLog('FlexCopFrontEnd Unknown voltage type.', $05);
{$ENDIF}
                        Result := -EINVAL;
                      end;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to bus voltage
                              . 13V (SEC_VOLTAGE_13)
                              . 18V (SEC_VOLTAGE_18)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end set voltage
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeSetVoltage2(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
begin
  try
    Result := IoCtlFeSetVoltage(FileDescriptor, Request, Parameter^);
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Enable higher voltage than usual (+0.5V)
                              . 0   (disabled)
                              . <>0 (enabled if suppoerted by hardware)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end enable high LNB voltage
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeEnableHighLnbVoltage(FileDescriptor: Integer; Request: Integer; Parameter: Integer): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_ENABLE_HIGH_LNB_VOLTAGE.', $04);
    ToLog('FlexCopFrontEnd Not supported.', $05);
{$ENDIF}
    Result := -EOPNOTSUPP;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to enable higher voltage than usual (+0.5V)
                              . 0   (disabled)
                              . <>0 (enabled if suppoerted by hardware)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end enable high LNB voltage
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeEnableHighLnbVoltage2(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
begin
  try
    Result := IoCtlFeEnableHighLnbVoltage(FileDescriptor, Request, Parameter^);
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to status result (word??)
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end read status
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeReadStatus(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
var
  Status            : Integer;
  StrengthPercentage: Double;
  StrengthDecibel   : Double;
  LockState         : Byte;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_READ_STATUS.', $05);
{$ENDIF}
    if not FlexCopQpskGetSignalPower(FileDescriptor, StrengthPercentage, StrengthDecibel) then
    begin
      Result := -EPERM;
      Exit;
    end;
    Status := 0;
    if StrengthPercentage > 10 then
      Status := Status or FE_HAS_SIGNAL;
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bVStatus, LockState) then
    begin
      Result := -EPERM;
      Exit;
    end;
    if (LockState and $80) <> 0 then
      Status := Status or FE_HAS_CARRIER;
    if (LockState and $10) <> 0 then
      Status := Status or FE_HAS_VITERBI;
    if (LockState and $08) <> 0 then
      Status := Status or FE_HAS_SYNC;
    if (LockState and $98) = $98 then
      Status := Status or FE_HAS_LOCK;
    Parameter^ := Status;
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd Status: $%8.8x.', [Status]), $05);
{$ENDIF}
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to where to store BER result
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end read bit error rate
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeReadBer(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
var
  Setting: Byte;
  Value  : Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_READ_BER.', $05);
{$ENDIF}
    Setting := CStv0299bDefaultsTBMU24112 [CStv0299bErrCnt, 1];
    // A setting of $93 means we have been setup for BER (otherwise uncorrected block count)
    if Setting = $93 then
    begin
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bECntH, Setting) then
      begin
        Result := -EPERM;
        Exit;
      end;
      Value := Setting shl 8;
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bECntL, Setting) then
      begin
        Result := -EPERM;
        Exit;
      end;
      Value := Value or Setting;
      Parameter^ := Value;
{$IFDEF USELOG}
      ToLog(format('FlexCopFrontEnd BER: %d.', [Value]), $05);
{$ENDIF}
      Result := ENOERROR;
    end
    else
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd Not supported.', $05);
{$ENDIF}
      Parameter^ := 0;
      Result     := -ENOSYS;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to where to store signal strength result
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end read signal strength
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeReadSignalStrength(FileDescriptor: Integer; Request: Integer; Parameter: PSmallInt): Integer;
var
  Strength: Integer;
  Data    : Byte;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_READ_SIGNAL_STRENGTH.', $05);
{$ENDIF}
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bAgc2I1, Data) then
    begin
      Result := -EPERM;
      Exit;
    end;
    Strength := Data shl 8;
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bAgc2I2, Data) then
    begin
      Result := -EPERM;
      Exit;
    end;
    Strength := Strength or Data;
    Strength := $FFFF - Strength;
    Strength := (Strength * 5) div 4;
    if Strength > $FFFF then
      Strength := $FFFF;
    if Strength < 0 then
      Strength := 0;
    Parameter^ := SmallInt(Strength);
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd Signal strength: %d.', [Strength]), $05);
{$ENDIF}
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to where to store SNR result
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end read signal noise ratio
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeReadSnr(FileDescriptor: Integer; Request: Integer; Parameter: PSmallInt): Integer;
var
  Noise: Integer;
  Data : Byte;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_READ_SNR.', $05);
{$ENDIF}
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bNirH, Data) then
    begin
      Result := -EPERM;
      Exit;
    end;
    Noise := Data shl 8;
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bNirL, Data) then
    begin
      Result := -EPERM;
      Exit;
    end;
    Noise := Noise or Data;
    Noise := $FFFF - Noise;
    Noise := Noise - $A100;
    Noise := Noise * 3;
    if Noise > $FFFF then
      Noise := $FFFF;
    if Noise < 0 then
      Noise := 0;
    Parameter^ := SmallInt(Noise);
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd SNR: %d.', [Noise]), $05);
{$ENDIF}
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to where to store uncorrected blocks result
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end read uncorrected blocks
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeReadUncorrectedBlocks(FileDescriptor: Integer; Request: Integer; Parameter: PInteger): Integer;
var
  Setting: Byte;
  Value  : Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_READ_UNCORRECTED_BLOCKS.', $05);
{$ENDIF}
    Setting := CStv0299bDefaultsTBMU24112[CStv0299bErrCnt, 0];
    // A setting of $B3 means we have been setup for uncorrected block count (else BER)
    if Setting = $B3 then
    begin
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bECntH, Setting) then
      begin
        Result := -EPERM;
        Exit;
      end;
      Value := Setting shl 8;
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bECntL, Setting) then
      begin
        Result := -EPERM;
        Exit;
      end;
      Value := Value or Setting;
      Parameter^ := Value;
{$IFDEF USELOG}
      ToLog(format('FlexCopFrontEnd Uncorrected blocks: %d.', [Value]), $05);
{$ENDIF}
      Result := ENOERROR;
    end
    else
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd Not supported.', $05);
{$ENDIF}
      Parameter^ := 0;
      Result     := -ENOSYS;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to front end parameters
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end set front end parameters
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeSetFrontEnd(FileDescriptor: Integer; Request: Integer; Parameter: PDvbFrontEndParameters): Integer;
const
  Reference = 4000000;
  Ratios : array[0..15] of Dword =
    (Reference div 2,  Reference div 4,   Reference div 8,   Reference div 16,
     Reference div 32, Reference div 64,  Reference div 128, Reference div 256,
     0,
     Reference div 5,  Reference div 10,  Reference div 20,  Reference div 40,
     Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency  =  950000;                      // Lowest frequency tuner (kHz)
  HighestTunerFrequency = 2150000;                      // Highest frequency tuner (kHz)
var
  Data                : Byte;
  CalcDivider         : Int64;
  ProgrammableDivider1: Byte;
  ProgrammableDivider2: Byte;
  ControlData1        : Byte;
  ControlData2        : Byte;
  SynthesizerData     : array[0..3] of Byte;
  Ratio               : Byte;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_SET_FRONTEND.', $04);
{$ENDIF}
    Result := -EINVAL;
    if Parameter.Inversion = INVERSION_OFF then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd INVERSION_OFF', $05);
{$ENDIF}
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bIoCfg, Data) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bIoCfg, Data and (not CStv0299bInversion)) then
        Exit;
    end
    else
      if Parameter.Inversion = INVERSION_ON then
      begin
{$IFDEF USELOG}
        ToLog('FlexCopFrontEnd INVERSION_ON', $05);
{$ENDIF}
        if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bIoCfg, Data) then
          Exit;
        if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bIoCfg, Data or CStv0299bInversion) then
          Exit;
      end
      else
      begin
        // Auto mode not supported, but generates no erro
{$IFDEF USELOG}
//        ToLog('FlexCopFrontEnd Inversion type not supported.', $05);
{$ENDIF}
//        Result := -EINVAL;
//        Exit;
      end;
      // Tune by setting manual mode; resetting derotator; setting auto mode
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bVSearch, CStv0299bManualSearchMode) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bCfrM, $00) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bCfrL, $00) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bVSearch, $19) then
        Exit;
      // Symbol rate
{$IFDEF USELOG}
      ToLog(format('FlexCopFrontEnd Symbol rate: %d.', [Parameter.Qpsk.SymbolRate]), $05);
{$ENDIF}
      // If the caller makes an error with the 'scale' of the symbolrate then
      // correct it
      if Parameter.Qpsk.SymbolRate > 1000000 then
      begin
        if not FlexCopQpskSetSymbolRate(FileDescriptor, Parameter.Qpsk.SymbolRate div 1000) then
          Exit;
      end
      else
      begin
        if not FlexCopQpskSetSymbolRate(FileDescriptor, Parameter.Qpsk.SymbolRate) then
          Exit;
      end;
      // FEC
      case Parameter.Qpsk.FecInner of
        FEC_1_2  : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_1_2.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $01) then
                       Exit;
                   end;
        FEC_2_3  : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_2_3.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $02) then
                       Exit;
                   end;
        FEC_3_4  : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_3_4.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $04) then
                       Exit;
                   end;
        FEC_5_6  : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_5_6.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $08) then
                       Exit;
                   end;
        FEC_7_8  : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_7_8.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $10) then
                       Exit;
                   end;
        FEC_AUTO : begin
{$IFDEF USELOG}
                     ToLog('FlexCopFrontEnd FEC_AUTO.', $05);
{$ENDIF}
                     if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bPr, $1F) then
                       Exit;
                   end;
        else
          begin
{$IFDEF USELOG}
            ToLog('FlexCopFrontEnd FEC mode not supported.', $05);
{$ENDIF}
            Result := -EINVAL;
            Exit;
          end;
      end;
{$IFDEF USELOG}
      ToLog(format('FlexCopFrontEnd Tuner frequency: %d.', [Parameter.Frequency]), $05);
{$ENDIF}
      // Frequency PLL
      if (Parameter.Frequency < LowestTunerFrequency) or
         (Parameter.Frequency > HighestTunerFrequency) then
      begin
        Result := -EINVAL;
        Exit;
      end;
      Ratio := 4;
      CalcDivider := Int64((Parameter.Frequency * 1000)) div Ratios[Ratio];
      ProgrammableDivider2 := CalcDivider and $FF;
      CalcDivider := CalcDivider shr 8;
      ProgrammableDivider1 := CalcDivider and $7F;

      ControlData1 := CSl1935DefaultControlRegister1 or Ratio;
      ControlData2 := CSl1935DefaultControlRegister2;
      // Depending on the frequency range we need to set the oscillator
      // 950-1500  or 1500-2150
      if Parameter.Frequency >= 1500000 then
        ControlData2 := ControlData2 or CSl1935SetOsc1500_2150
      else
        ControlData2 := ControlData2 or CSl1935SetOsc950_1500;

      SynthesizerData[0] := ProgrammableDivider1;
      SynthesizerData[1] := ProgrammableDivider2;
      SynthesizerData[2] := ControlData1;
      SynthesizerData[3] := ControlData2;
      // Now we can send the data to the synthesizer
      if not FlexCopWriteToSynthesizer(FileDescriptor, 1, SynthesizerData) then
        Exit;
      // Because the STV0299 might have to lock to this (new) frequency we have
      // to reset the de-rotator loop
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bCfrM, $00) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bCfrL, $00) then
        Exit;
      if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bCfrL, Data) then
        Exit;
      if not FlexCopWriteToQpsk(FileDescriptor, CStv0299bCfd, $B9) then
        Exit;

      Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to front end parameters
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end get front end parameters
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeGetFrontEnd(FileDescriptor: Integer; Request: Integer; Parameter: PDvbFrontEndParameters): Integer;
var
  Data      : Byte;
  SymbolRate: Dword;
  Calc      : Extended;
  Deviation : Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_GET_FRONTEND.', $04);
{$ENDIF}
    Result := -EFAULT;
    // Inversion
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bIoCfg, Data) then
      Exit;
    if (Data and CStv0299bInversion) = 0 then
      Parameter^.Inversion := INVERSION_OFF
    else
      Parameter^.Inversion := INVERSION_ON;
    // FEC
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bVStatus, Data) then
      Exit;
    case (Data and $07) of
      0: Parameter^.Qpsk.FecInner := FEC_2_3;
      1: Parameter^.Qpsk.FecInner := FEC_3_4;
      2: Parameter^.Qpsk.FecInner := FEC_5_6;
      3: Parameter^.Qpsk.FecInner := FEC_7_8;
      4: Parameter^.Qpsk.FecInner := FEC_1_2;
      else
         Parameter^.Qpsk.FecInner := FEC_AUTO;
    end;
    // Symbol rate
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bSfrH, Data) then
      Exit;
    SymbolRate := Data shl 16;
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bSfrM, Data) then
      Exit;
    SymbolRate := SymbolRate or (Data shl 8);
    if not FlexCopReadFromQpsk(FileDescriptor, CStv0299bSfrL, Data) then
      Exit;
    SymbolRate := SymbolRate or (Data and $F0);
    SymbolRate := SymbolRate shr 4;
    Calc := SymbolRate;
    Calc := Calc * CStv0299bSetSymbolrateUnits;
    Parameter^.Qpsk.SymbolRate := Round(Calc);
    // Frequency deviation
    if not FlexCopQpskGetCarrierDeviation(FileDescriptor, Deviation) then
      Exit;
    Parameter^.Frequency := Parameter^.Frequency + Deviation;
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to front end event
  Returns : <Result>          Result code

  Descript: IoCtl function: Front end get event
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlFeGetEvent(FileDescriptor: Integer; Request: Integer; Parameter: PDvbFrontEndEvent): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog('FlexCopFrontEnd FE_GET_EVENT.', $04);
    ToLog('FlexCopFrontEnd Not supported.', $05);
{$ENDIF}
    Result := -EOPNOTSUPP;
  except
    Result := -EFAULT;
  end;
end;
{******************************************************************************}
{                            LINUX API FUNCTIONS END                           }
{******************************************************************************}


{******************************************************************************}
{                       EXPORTED LINUX API FUNCTIONS                           }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <DeviceName>  Name of specific video device
            <Flags>       Flags
  Returns : <Result>      File descriptor (handle) or result code

  Descript: Open function
  Notes   :
 ------------------------------------------------------------------------------}
function Open(const DeviceName: PChar; const Flags: Integer): Integer; stdcall;
var
  CheckName : string;
  Number    : Char;
  CardNumber: Byte;
  Error     : Integer;
  CardHandle: THandle;
begin
  try
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd ''Open'', Device name ''%s''.', [DeviceName]), $03);
{$ENDIF}
    // The name of the device, or more precisely the trailing number, indicates
    // which card to use. We ony support a singl number here
    CheckName := DeviceName;
    if CheckName = '' then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''Open'', Error: ''No device name specified''.', $02);
{$ENDIF}
      Result := -EINVAL;
      Exit;
    end;
    Number := CheckName[Length(CheckName)];
    Val(Number, CardNumber, Error);
    if Error <> 0 then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''Open'', Error: ''Invalid device name number''.', $02);
{$ENDIF}
      Result := -EINVAL;
      Exit;
    end;
    CardHandle := FlexCopCreateFile(CardNumber);
    if CardHandle = INVALID_HANDLE_VALUE then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''Open'', Error: ''No such device''.', $02);
{$ENDIF}
      Result := -ENODEV
    end
    else
    begin
      // We also have to initialize the tuner to the default state
      if not FlexCopWriteDefaultsToQpsk(CardHandle, TunerType) then
      begin
        FlexCopCloseHandle(CardHandle);
{$IFDEF USELOG}
        ToLog('FlexCopFrontEnd ''Open'', Error: ''Unable to access QPSK''.', $02);
{$ENDIF}
        Result := -EFAULT;
      end
      else
      begin
        // Power on (if supported)
        if not FlexCopEnableLNB(CardHandle) then
        begin
{$IFDEF USELOG}
          ToLog('FlexCopFrontEnd ''Open'', Error: ''Unable to enable LNB''.', $02);
{$ENDIF}
          Result := -EFAULT
        end
        else
          Result := CardHandle;
      end;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
  Returns : <Result>          Result code

  Descript: Close function
  Notes   :
 ------------------------------------------------------------------------------}
function Close(const FileDescriptor: Integer): Integer; stdcall;
begin
  try
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd ''Close'', File descriptor %d.', [FileDescriptor]), $03);
{$ENDIF}
    if FileDescriptor < 0 then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''Close'', Error: ''Bad file number''.', $02);
{$ENDIF}
      Result := -EBADF;
      Exit;
    end;
    try
      FlexCopDisableLNB(FileDescriptor);
      FlexCopCloseHandle(FileDescriptor);
      Result := ENOERROR;
    except
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''Close'', Error: ''Exception occured''.', $02);
{$ENDIF}
      Result := -EBADF;
    end;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to parameter
  Returns : <Result>          Result code

  Descript: IoCtl function entry
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtl(const FileDescriptor: Integer; const Request: Integer; Parameter: Pointer): Integer; stdcall;
var
  IoCtlIndex: Integer;
begin
  try
{$IFDEF USELOG}
    ToLog(format('FlexCopFrontEnd ''IoCtl'', File descriptor $%x, Request $%8.8x.', [FileDescriptor, Request]), $05);
{$ENDIF}
    if Dword(FileDescriptor) = INVALID_HANDLE_VALUE then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''IoCtl'', Error: ''Bad file number''.', $02);
{$ENDIF}
      Result := -EBADF;
      Exit;
    end;
    // Find the function belonging to the <Request>
    IoCtlIndex := Low(IoCtlIdentifier);
    while (IoCtlIdentifier[IoCtlIndex] <> Request) and (IoCtlIndex <= High(IoCtlIdentifier)) do
      Inc(IoCtlIndex);
    // No identifier found means we don't support it  
    if IoCtlIndex > High(IoCtlIdentifier) then
    begin
{$IFDEF USELOG}
      ToLog('FlexCopFrontEnd ''IoCtl'', Error: ''Function not supported''.', $02);
{$ENDIF}
      Result := -EOPNOTSUPP;
      Exit;
    end;

    // Check for correct <Parameter>
    case IoCtlIndex of
      Ord(IoCtl_FE_DISEQC_RESET_OVERLOAD),
      Ord(IoCtl_FE_DISEQC_SEND_BURST),
      Ord(IoCtl_FE_SET_TONE),
      Ord(IoCtl_FE_SET_VOLTAGE),
      Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE): ;
      else
        begin
          if not Assigned(Parameter) then
          begin
{$IFDEF USELOG}
            ToLog('FlexCopFrontEnd ''IoCtl'', Error: ''Bad address''.', $02);
{$ENDIF}
            Result := -EFAULT;
            Exit;
          end;
        end;
    end;
    case IoCtlIndex of
      Ord(IoCtl_FE_GET_INFO                ): Result := IoCtlFeGetInfo              (FileDescriptor, Request, PDvbFrontEndInfo(Parameter));
      Ord(IoCtl_FE_DISEQC_RESET_OVERLOAD   ): Result := IoCtlFeDiSEqCResetOverload  (FileDescriptor, Request);
      Ord(IoCtl_FE_DISEQC_SEND_MASTER_CMD  ): Result := IoCtlFeDiSEqCSendMasterCmd  (FileDescriptor, Request, PDvbDiSEqCMasterCmd(Parameter));
      Ord(IoCtl_FE_DISEQC_RECV_SLAVE_REPLY ): Result := IoCtlFeDiSEqCRecvSlaveReply (FileDescriptor, Request, PDvbDiSEqCSlaveReply(Parameter));
      Ord(IoCtl_FE_DISEQC_SEND_BURST       ): Result := IoCtlFeDiSEqCSendBurst      (FileDescriptor, Request, Integer(Parameter));

      Ord(IoCtl_FE_SET_TONE                ): Result := IoCtlFeSetTone              (FileDescriptor, Request, Integer(Parameter));
      Ord(IoCtl_FE_SET_VOLTAGE             ): Result := IoCtlFeSetVoltage           (FileDescriptor, Request, Integer(Parameter));
      Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE ): Result := IoCtlFeEnableHighLnbVoltage (FileDescriptor, Request, Integer(Parameter));

      Ord(IoCtl_FE_READ_STATUS             ): Result := IoCtlFeReadStatus           (FileDescriptor, Request, PInteger(Parameter));
      Ord(IoCtl_FE_READ_BER                ): Result := IoCtlFeReadBer              (FileDescriptor, Request, PInteger(Parameter));
      Ord(IoCtl_FE_READ_SIGNAL_STRENGTH    ): Result := IoCtlFeReadSignalStrength   (FileDescriptor, Request, PSmallInt(Parameter));
      Ord(IoCtl_FE_READ_SNR                ): Result := IoCtlFeReadSnr              (FileDescriptor, Request, PSmallInt(Parameter));
      Ord(IoCtl_FE_READ_UNCORRECTED_BLOCKS ): Result := IoCtlFeReadUncorrectedBlocks(FileDescriptor, Request, PInteger(Parameter));

      Ord(IoCtl_FE_SET_FRONTEND            ): Result := IoCtlFeSetFrontEnd          (FileDescriptor, Request, PDvbFrontEndParameters(Parameter));
      Ord(IoCtl_FE_GET_FRONTEND            ): Result := IoCtlFeGetFrontEnd          (FileDescriptor, Request, PDvbFrontEndParameters(Parameter));
      Ord(IoCtl_FE_GET_EVENT               ): Result := IoCtlFeGetEvent             (FileDescriptor, Request, PDvbFrontEndEvent(Parameter));
      // Additional functions
      Ord(IoCtl_FE_DISEQC_SEND_BURST2      ): Result := IoCtlFeDiSEqCSendBurst2     (FileDescriptor, Request, PInteger(Parameter));
      Ord(IoCtl_FE_SET_TONE2               ): Result := IoCtlFeSetTone2             (FileDescriptor, Request, PInteger(Parameter));
      Ord(IoCtl_FE_SET_VOLTAGE2            ): Result := IoCtlFeSetVoltage2          (FileDescriptor, Request, PInteger(Parameter));
      Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE2): Result := IoCtlFeEnableHighLnbVoltage2(FileDescriptor, Request, PInteger(Parameter));
      else
        begin
          // We sould NEVER get here ....
{$IFDEF USELOG}
          ToLog('FlexCopFrontEnd ''IoCtl'', Error: ''Function not supported''.', $02);
{$ENDIF}
          Result := -EOPNOTSUPP;
        end;
    end;
  except
    Result := -EFAULT;
  end;
end;
{******************************************************************************}
{                       EXPORTED LINUX API FUNCTIONS END                       }
{******************************************************************************}


{******************************************************************************}
{                              INITIALIZE AND FINALIZE                         }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  Parameter    : string;
  Error        : Integer;
{$IFDEF USELOG}
  LogClear     : Boolean;
{$ENDIF}
  Loop         : Integer;
  Value        : Integer;
begin
  // Set initial values
  IniFile      := nil;
{$IFDEF USELOG}
  LogStream    := nil;
  LogLevel     := 0;
  LogClear     := True;
{$ENDIF}

  for Loop := Low(IoCtlIdentifier) to High(IoCtlIdentifier) do
    IoCtlIdentifier[Loop] := -1;

  if FileExists('FlexCop.ini') then
  begin
    IniFile := TMemIniFile.Create('FlexCop.ini');
{$IFDEF USELOG}
    // Only with a local INI file we allow logging
    Parameter := GetParameter('FrontEnd', 'LogLevel', '0');
    Val(Parameter, LogLevel, Error);
    if Error <> 0 then
      LogLevel := 0;
    Parameter := GetParameter('FrontEnd', 'LogClear', 'Yes');
    if LowerCase(Parameter) = 'no' then
      LogClear := False
    else
      LogClear := True;
{$ENDIF}
  end;
{$IFDEF USELOG}
  if LogLevel in [1..5] then
  begin
    try
      if (not LogClear) and
         FileExists('FlexCopFrontEnd.log') then
      begin
        LogStream := TFileStream.Create('FlexCopFrontEnd.log', fmOpenReadWrite);
        // Go to end of file (we will be appending)
        LogStream.Seek(0, soFromEnd);
      end
      else
        LogStream := TFileStream.Create('FlexCopFrontEnd.log', fmCreate);
      ToLog('-------------------------------------------------------------', $00);
      ToLog(format('Log level: %d', [LogLevel]), $00);
    except
      FreeAndNil(LogStream);
    end;
  end;
{$ENDIF}

  // Get IoCtl identifiers
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_GET_INFO', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_GET_INFO)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_DISEQC_RESET_OVERLOAD', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_DISEQC_RESET_OVERLOAD)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_DISEQC_SEND_MASTER_CMD', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_DISEQC_SEND_MASTER_CMD)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_DISEQC_RECV_SLAVE_REPLY', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_DISEQC_RECV_SLAVE_REPLY)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_DISEQC_SEND_BURST', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_DISEQC_SEND_BURST)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_SET_TONE', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_SET_TONE)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_SET_VOLTAGE', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_SET_VOLTAGE)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_ENABLE_HIGH_LNB_VOLTAGE', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_READ_STATUS', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_READ_STATUS)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_READ_BER', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_READ_BER)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_READ_SIGNAL_STRENGTH', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_READ_SIGNAL_STRENGTH)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_READ_SNR', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_READ_SNR)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_READ_UNCORRECTED_BLOCKS', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_READ_UNCORRECTED_BLOCKS)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_SET_FRONTEND', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_SET_FRONTEND)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_GET_FRONTEND', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_GET_FRONTEND)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_GET_EVENT', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_GET_EVENT)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_DISEQC_SEND_BURST2', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_DISEQC_SEND_BURST2)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_SET_TONE2', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_SET_TONE2)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_SET_VOLTAGE2', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_SET_VOLTAGE2)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'FE_ENABLE_HIGH_LNB_VOLTAGE2', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_FE_ENABLE_HIGH_LNB_VOLTAGE2)] := Value;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
{$IFDEF USELOG}
  ToLog('FlexCopFrontEnd Finalizing.', $03);
  if Assigned(LogStream) then
    FreeAndNil(LogStream);
{$ENDIF}
  if Assigned(IniFile) then
    FreeAndNil(IniFile);
end;


initialization
  Initialize;

  finalization
    Finalize;
end.

