//{ **************************************************************************** }
//{ FileName............: REMOVELOCK.CPP                                         }
//{ Project.............: Generic                                                }
//{ Author(s)...........: M.Majoor                                               }
//{ Original contents...: Walter Oney ('Programming the Windows Driver Model')   }
//{ Version.............: 3.20                                                   }
//{ ---------------------------------------------------------------------------- }
//{ Portable implementation for remove-lock functions.                           }
//{                                                                              }
//{ Version  Date      Comment                                                   }
//{ 1.00     20050410  - Initial release based on SAA7146A driver V2.02          }
//{ 2.10     20080126  - No changes                                              }
//{ 3.20     20171213  - Initialization values and macros added to prevent       }
//{                      compiler warnings                                       }
//{ **************************************************************************** }


#include "stddcls.h"


//{ **************************************************************************** }
//{ Params   : <lock>      Lock                                                  }
//{            <tag>       ..                                                    }
//{            <minutes>   Max time for lock                                     }
//{            <maxcount>  Maximum locks                                         }
//{ Return   : -                                                                 }
//{                                                                              }
//{ Descript : Initialize remove lock.                                           }
//{ Notes    :                                                                   }
//{ **************************************************************************** }
#pragma PAGEDCODE
VOID InitializeRemoveLock(PREMOVE_LOCK lock, ULONG tag, ULONG minutes, ULONG maxcount)
{
  UNREFERENCED_PARAMETER(maxcount);
  UNREFERENCED_PARAMETER(minutes);
  UNREFERENCED_PARAMETER(tag);
  PAGED_CODE();
  KeInitializeEvent(&lock->evRemove, NotificationEvent, FALSE);
  lock->usage = 1;
  lock->removing = FALSE;
}


//{ **************************************************************************** }
//{ Params   : <lock>    Lock                                                    }
//{            <tag>     ..                                                      }
//{ Return   : <Result>  NTSTATUS                                                }
//{                                                                              }
//{ Descript : Acquire remove lock.                                              }
//{ Notes    :                                                                   }
//{ **************************************************************************** }
#pragma LOCKEDCODE
NTSTATUS AcquireRemoveLock(PREMOVE_LOCK lock, PVOID tag)
{
  UNREFERENCED_PARAMETER(tag);
  LONG usage = InterlockedIncrement(&lock->usage);         // Adjust usage counter
  UNREFERENCED_PARAMETER(usage);
  if (lock->removing)
  {                                                        // If removal in progress
    if (InterlockedDecrement(&lock->usage) == 0)
      KeSetEvent(&lock->evRemove, 0, FALSE);
    return STATUS_DELETE_PENDING;
  }
  return STATUS_SUCCESS;
}


//{ **************************************************************************** }
//{ Params   : <lock>    Lock                                                    }
//{            <tag>     ..                                                      }
//{ Return   : -                                                                 }
//{                                                                              }
//{ Descript : Release acquired remove lock.                                     }
//{ Notes    :                                                                   }
//{ **************************************************************************** }
#pragma LOCKEDCODE
VOID ReleaseRemoveLock(PREMOVE_LOCK lock, PVOID tag)
{
  UNREFERENCED_PARAMETER(tag);
  if (InterlockedDecrement(&lock->usage) == 0)
    KeSetEvent(&lock->evRemove, 0, FALSE);
}


//{ **************************************************************************** }
//{ Params   : <lock>    Lock                                                    }
//{            <tag>     ..                                                      }
//{ Return   : -                                                                 }
//{                                                                              }
//{ Descript : Release acquired remove lock and wait for it to be released.      }
//{ Notes    :                                                                   }
//{ **************************************************************************** }
#pragma PAGEDCODE
VOID ReleaseRemoveLockAndWait(PREMOVE_LOCK lock, PVOID tag)
{
  PAGED_CODE();
  lock->removing = TRUE;
  ReleaseRemoveLock(lock, tag);
  ReleaseRemoveLock(lock, NULL);
  KeWaitForSingleObject(&lock->evRemove, Executive, KernelMode, FALSE, NULL);
}
