{******************************************************************************}
{ FileName............: MajorDvbPsiUnit001                                     }
{ Project.............: DirectShow                                             }
{ Author(s)...........: MM                                                     }
{ Version.............: 2.01                                                   }
{------------------------------------------------------------------------------}
{  DirectShow PSI parsing filter                                               }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  2.00   20040202 - Initial release                                           }
{  2.01   20040312 - Added additional debug counters                           }
{                  - Corrected incorrect call to <DvbGetPid>                   }
{                  - Data size is from <Sample.GetActualDataLength>            }
{                  - Added command to return latched data and initates new     }
{                    latch of 512 bytes of data                                }
{                  - Added command to write buffers to files                   }
{                  - <IID_ISpecifyPropertyPages> used in                       }
{                    <NonDelegatingQueryInterface>                             }
{                  - Added About property page                                 }  
{******************************************************************************}
unit MajorDvbPsiUnit001;

interface
uses
  ActiveX,
  BaseClass,
  Classes,
  DirectShow9,
  DvbFilter,
  Messages,
  SysUtils,
  Windows;


const
  Version = $0201;
  Build   = $20040312;

  IID_IPersistStream                 : TGUID = '{0000010C-0000-0000-C000-000000000046}';
  IID_ISpecifyPropertyPages          : TGUID = '{B196B28B-BAB4-101A-B69C-00AA00341D07}';

  // Interfaces for the outside world
  IID_MajorDvbPsi                            = '{6D616A6F-7269-7479-6E6C-707369020201}';
  T_IID_MajorDvbPsi                  : TGUID = IID_MajorDvbPsi;

  // Classes
  CLSID_MajorDvbPsi                  : TGUID = '{6D616A6F-7269-7479-6E6C-707369000201}';
  CLSID_MajorDvbPsiPropertyPage      : TGUID = '{6D616A6F-7269-7479-6E6C-707369010201}';
  CLSID_MajorDvbPsiPropertyPageAbout : TGUID = '{6D616A6F-7269-7479-6E6C-707369FF0201}';

  SudPinTypes : TRegPinTypes =
    (clsMajorType: @MEDIATYPE_NULL;
     clsMinorType: @MEDIASUBTYPE_NULL);

  SudPins : array[0..1] of TRegFilterPins =
    ((strName: 'Input' ; bRendered: FALSE; bOutput: FALSE; bZero: FALSE; bMany: FALSE; oFilter: nil; strConnectsToPin: ''; nMediaTypes: 1; lpMediaType: @SudPinTypes),
     (strName: 'Output'; bRendered: FALSE; bOutput: TRUE;  bZero: FALSE; bMany: FALSE; oFilter: nil; strConnectsToPin: ''; nMediaTypes: 1; lpMediaType: @SudPinTypes));


type
{------------------------------------------------------------------------------
  Descript: External access interface
 ------------------------------------------------------------------------------}
  IMajorDvbPsi = interface(IUnknown)
  [IID_MajorDvbPsi]
    // Debug / informational
    function  GetVersionInformation(out Info: PChar): HRESULT; stdcall;
    function  GetDebugInformation  (out Info: PChar): HRESULT; stdcall;
    function  GetDebugCount        (out Info: Word) : HRESULT; stdcall;
    function  GetDebugCounters     (out DsFilterIn: Word; out DsFilterInSizeCorrect: Word; out Packets: Word; out ValidSyncs: Word; out FalseSyncs: Word; out ToPidFilter: Word) : HRESULT; stdcall;
    function  GetDebugBufferSize   (out Info: LongWord) : HRESULT; stdcall;
    function  GetDebugValidDataSize(out Info: LongWord) : HRESULT; stdcall;
    function  GetDebugBufferData   (out Info: Word; Buffer: Pointer) : HRESULT; stdcall;
    function  SetDebugWriteBuffer  (Buffers: Byte) : HRESULT; stdcall;
    procedure SetDebugSetLog       (OnOff: Boolean); stdcall;

    // PSI related
    function  DvbGetProgramName(ToGet: Word; out Info: PChar)    : HRESULT; stdcall;
    function  DvbGetPid(StreamPacketData: PDvbTransportPacket; out Pid: Word): Boolean; stdcall;
    function  DvbSetSignallingCallback     (Pid: Word; hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetSignallingCallbackPmt  (hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean; stdcall;
    function  DvbGetPidFilter(Pid: Word): TDvbPacketFilter; stdcall;
    function  DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean; stdcall;
    function  DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter; stdcall;
    function  DvbGetProgramInfo(ProgramNumber: Byte;
                                out PmtPid: Word;
                                out PcrPid: Word;
                                out VideoPids: TDvbPids;
                                out AudioPids: TDvbPids;
                                out TeletextPids: TDvbPids;
                                out SubtitlePids: TDvbPids;
                                out AudioLanguages: TDvbLanguages;
                                out SubtitleLanguages: TDvbLanguages;
                                out EcmPids: TDvbPids;
                                out CaIds: TDvbPids;
                                out ProgramName: PChar): Boolean; stdcall;
    function  DvbGetEventInfo(ProgramNumber: Byte; Present: Boolean; Events: TDvbEventSections ): Boolean; stdcall;
    function  DvbGetNumberOfPrograms: Byte; stdcall;
    function  DvbGetErrors: Word; stdcall;
    function  DvbGetPacketSyncErrors: Word; stdcall;
    procedure DvbResetErrors; stdcall;
    procedure DvbCreateTables; stdcall;
  end;


{------------------------------------------------------------------------------
  Descript: Input pin
 ------------------------------------------------------------------------------}
  TMajorDvbPsiInputPin = class(TBCTransInPlaceInputPin)
  public
    constructor Create(ObjectName: string; TransInPlaceFilter: TBCTransInPlaceFilter;out hr: HRESULT; Name: WideString);
  end;


{------------------------------------------------------------------------------
  Descript: Output pin
 ------------------------------------------------------------------------------}
  TMajorDvbPsiOutputPin = class(TBCTransInPlaceOutputPin)
  public
    constructor Create(ObjectName: string; TransInPlaceFilter: TBCTransInPlaceFilter;out hr: HRESULT; Name: WideString);
  end;


{------------------------------------------------------------------------------
  Descript: The filter
  Notes   : The <IPersistStream> is required to make the filter 'savable'
            and 'loadable' (eg. by graphedit)
            The <IPersistStream> retains our settings.
 ------------------------------------------------------------------------------}
  TMajorDvbPsi = class(TBCTransInPlaceFilter, IMajorDvbPsi, ISpecifyPropertyPages, IPersistStream)
  private
    FThisInstance    : Integer;
    PsiLock          : TBCCritSec;
    FDebugInformation: string;
    FDebugCountDsFilterIn           : Word;                // Data entering filter
    FDebugCountDsFilterInSizeCorrect: Word;                // Data entering filter has correct size
    FDebugCountPackets              : Word;                // Packets
    FDebugCountValidSyncs           : Word;                // Packet with valid sync byte
    FDebugCountFalseSyncs           : Word;                // Packet with false sync byte
    FDebugCountToPidFilter          : Word;                // Packet with PID going into filter
    FBufferSize      : LongWord;                           // Size of buffer
    FValidDataSize   : LongWord;                           // Size of valid data in buffer
    FLatchedData     : packed array[0..511] of Byte;       // Latched data
    FLatchData       : Boolean;                            // Set TRUE when new latching should take place
                                                           // Set to FALSE when latching done
    FFileBuffers     : Byte;                               // Set to the number of buffers to write to a file
                                                           // Set to 0 when writing done
    FFilename        : string;                             // Output file name
    FFileBuffersMax  : Byte;                               // Copy of initial <FFileBuffers> for index

    FLogStream       : TFileStream;                        // Filestream for debug log
  protected
  public
    constructor Create(ObjName: string; unk: IUnknown; out hr: HRESULT);
    constructor CreateFromFactory(Factory: TBCClassFactory; const Controller: IUnknown); override;
    destructor  Destroy; override;

    // Required methods to implement (TransInPlace)
    function    CheckInputType(MediaTypeIn: PAMMediaType): HRESULT; override;
    function    Transform(Sample: IMediaSample): HRESULT; override;

    // <ISpecifyPropertyPages>
    function    GetPages(out Pages: TCAGUID): HRESULT;    stdcall;

    // <IPersistStream>
    function    IsDirty: HRESULT; stdcall;
    function    Load(const stm: IStream): HRESULT; stdcall;
    function    Save(const stm: IStream; fClearDirty: BOOL): HRESULT; stdcall;
    function    GetSizeMax(out cbSize: Largeint): HRESULT; stdcall;

    // Specials
    function    NonDelegatingQueryInterface(const IID: TGUID; out Obj): HRESULT; override;

    function  GetVersionInformation(out Info: PChar): HRESULT; stdcall;
    function  GetDebugInformation  (out Info: PChar): HRESULT; stdcall;
    function  GetDebugCount        (out Info: Word) : HRESULT; stdcall;
    function  GetDebugCounters     (out DsFilterIn: Word; out DsFilterInSizeCorrect: Word; out Packets: Word; out ValidSyncs: Word; out FalseSyncs: Word; out ToPidFilter: Word) : HRESULT; stdcall;
    function  GetDebugBufferSize   (out Info: LongWord) : HRESULT; stdcall;
    function  GetDebugValidDataSize(out Info: LongWord) : HRESULT; stdcall;
    function  GetDebugBufferData   (out Info: Word; Buffer: Pointer) : HRESULT; stdcall;
    function  SetDebugWriteBuffer  (Buffers: Byte) : HRESULT; stdcall;
    procedure SetDebugSetLog       (OnOff: Boolean); stdcall;

    // PSI specific functions
    function  DvbGetProgramName(ToGet: Word; out Info: PChar)    : HRESULT; stdcall;
    function  DvbGetPid(StreamPacketData: PDvbTransportPacket; out Pid: Word): Boolean; stdcall;
    function  DvbSetSignallingCallback     (Pid: Word; hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetSignallingCallbackPmt  (hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean; stdcall;
    function  DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean; stdcall;
    function  DvbGetPidFilter(Pid: Word): TDvbPacketFilter; stdcall;
    function  DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean; stdcall;
    function  DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter; stdcall;
    function  DvbGetProgramInfo(ProgramNumber: Byte;
                                out PmtPid: Word;
                                out PcrPid: Word;
                                out VideoPids: TDvbPids;
                                out AudioPids: TDvbPids;
                                out TeletextPids: TDvbPids;
                                out SubtitlePids: TDvbPids;
                                out AudioLanguages: TDvbLanguages;
                                out SubtitleLanguages: TDvbLanguages;
                                out EcmPids: TDvbPids;
                                out CaIds: TDvbPids;
                                out ProgramName: PChar): Boolean; stdcall;
    function  DvbGetEventInfo(ProgramNumber: Byte; Present: Boolean; Events: TDvbEventSections ): Boolean; stdcall;
    function  DvbGetNumberOfPrograms: Byte; stdcall;
    function  DvbGetErrors: Word; stdcall;
    function  DvbGetPacketSyncErrors: Word; stdcall;
    procedure DvbResetErrors; stdcall;
    procedure DvbCreateTables; stdcall;

    procedure ToLog(Logstring: string);
  published
  end;

var
  InstanceCount : Integer;


implementation


{------------------------------------------------------------------------------
  Params  : <LogString>  Text to output to log file
  Returns : -

  Descript: Put text to log file
  Notes   : Also sets <FDebugInformation>
 ------------------------------------------------------------------------------}
procedure TMajorDvbPsi.ToLog(LogString: string);
var
  NewLog: string;
begin
  if not Assigned(FLogStream) then
    Exit;
  NewLog := FormatDateTime('YYYYMMDD"T"HHMMSS"  "', Now) + LogString + #13#10;
  FLogStream.Write(NewLog[1], Length(NewLog));
  FDebugInformation := LogString;
end;


{------------------------------------------------------------------------------
  Params  : <ObjectName>
            <TransInPlaceFilter>
            <Name>
  Returns : <hr>

  Descript: Create the input pin
  Notes   :
 ------------------------------------------------------------------------------}
constructor TMajorDvbPsiInputPin.Create(ObjectName: string;
  TransInPlaceFilter: TBCTransInPlaceFilter; out hr: HRESULT;
  Name: WideString);
begin
  inherited Create(ObjectName, TransInPlaceFilter, hr, Name);
end;


{------------------------------------------------------------------------------
  Params  : <ObjectName>
            <TransInPlaceFilter>
            <Name>
  Returns : <hr>

  Descript: Create the output pin.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TMajorDvbPsiOutputPin.Create(ObjectName: string;
  TransInPlaceFilter: TBCTransInPlaceFilter; out hr: HRESULT;
  Name: WideString);
begin
  inherited Create(ObjectName, TransInPlaceFilter, hr, Name);
end;


{------------------------------------------------------------------------------
  Params  : <ObjName>
            <Unk>
  Returns : <hr>

  Descript: Create filter.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TMajorDvbPsi.Create(ObjName: string; unk: IUnknown;
  out hr: HRESULT);
begin
  // The ending 'FALSE' is the modify flag. We indicate we don't modify the data.
  inherited Create(ObjName, unk, CLSID_MajorDvbPsi, hr, False);

  FDebugInformation := 'No debug information';
  FDebugCountDsFilterIn            := 0;
  FDebugCountDsFilterInSizeCorrect := 0;
  FDebugCountPackets               := 0;
  FDebugCountValidSyncs            := 0;
  FDebugCountFalseSyncs            := 0;
  FDebugCountToPidFilter           := 0;

  FBufferSize      := 0;
  FValidDataSize   := 0;
  FLatchData       := False;
  FFileBuffers     := 0;

  FLogStream       := nil;

  FThisInstance := InterlockedIncrement(InstanceCount);

  PsiLock   := TBCCritSec.Create;
end;


{------------------------------------------------------------------------------
  Params  : <Factory>
            <COntroller>
  Returns : -

  Descript: Create filter 'from factory'.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TMajorDvbPsi.CreateFromFactory(Factory: TBCClassFactory;
  const Controller: IUnknown);
var
  hr: HRESULT;
begin
  Create(Factory.Name, Controller, hr);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor.
  Notes   :
 ------------------------------------------------------------------------------}
destructor TMajorDvbPsi.Destroy;
begin
  // Release all created instances
  FLogStream.Free;
  PsiLock.Free;
  inherited;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Result
            <Pages>   Property pages GUID

  Descript: Get property pages.
  Notes   : <ISpecifyPropertyPages>
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetPages(out pages: TCAGUID): HRESULT;
begin
  ToLog('<GetPages>');
  Pages.cElems := 2;
  Pages.pElems := CoTaskMemAlloc(SizeOf(TGUID) * Pages.cElems);
  if not Assigned(Pages.pElems) then
  begin
    FDebugInformation := '<GetPages> failed';
    Result := E_OUTOFMEMORY;
    ToLog('<FetPages> E_OUTOFMEMORY');
    Exit;
  end;
  Pages.pElems^[0] := CLSID_MajorDvbPsiPropertyPage;
  Pages.pElems^[1] := CLSID_MajorDvbPsiPropertyPageAbout;
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : <MediaTypeIn>  Media type of the input
  Returns : <Result>       S_OK if acceptable

  Descript: Check input media type
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.CheckInputType(MediaTypeIn: PAMMediaType): HRESULT;
var
  Accepted: Boolean;
begin
  Accepted := True;
//  Accepted := False;
//  // Accept only streams
//  if IsEqualGUID(MediaTypeIn.MajorType, MEDIATYPE_STREAM) then
//    Accepted := True;

  if Accepted = True then
    Result := S_OK
  else
    Result := VFW_E_TYPE_NOT_ACCEPTED;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <Info>    Informational data

  Descript: Get version information data.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetVersionInformation(out Info: PChar): HRESULT;
begin
  Result := S_FALSE;
  if not Assigned(Info) then
    Exit;
  StrPCopy(Info, format('V%1.1x.%2.2x, build %x', [Version div $100, Version mod $100, Build]));
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <Info>    Informational data

  Descript: Get debug data.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugInformation(out Info: PChar): HRESULT;
begin
  Result := S_FALSE;
  if not Assigned(Info) then
    Exit;
  StrPCopy(Info, FDebugInformation);
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <Info>    Debug counter

  Descript: Get debug data.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugCount(out Info: Word): HRESULT;
begin
  Info := FDebugCountDsFilterIn;
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <DsFilterIn>             Debug counter (data entering filter)
            <DsFilterInSizeCorrect>  Debug counter (valid data size detected)
            <Packets>                Debug counter (packets)
            <ValidSyncs>             Debug counter (valid packet == sync found)
            <FalseSyncs>             Debug counter (valid packet == sync found)
            <ToPidFilter>            Debug counter (filtered packets NOT necessarily correct data)

  Descript: Get debug data.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugCounters(out DsFilterIn: Word; out DsFilterInSizeCorrect: Word; out Packets: Word; out ValidSyncs: Word; out FalseSyncs: Word; out ToPidFilter: Word) : HRESULT; stdcall;
begin
  DsFilterIn            := FDebugCountDsFilterIn;
  DsFilterInSizeCorrect := FDebugCountDsFilterInSizeCorrect;
  Packets               := FDebugCountPackets;
  ValidSyncs            := FDebugCountValidSyncs;
  FalseSyncs            := FDebugCountFalseSyncs;
  ToPidFilter           := FDebugCountToPidFilter;
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <Info>    Last detected buffer size

  Descript: Get buffer size.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugBufferSize(out Info: LongWord): HRESULT;
begin
  Info := FBufferSize;
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_OK
            <Info>    Last detected valid data size

  Descript: Get valid data size.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugValidDataSize(out Info: LongWord): HRESULT;
begin
  Info := FValidDataSize;
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>  Target buffer (must have room for at least 512 bytes)
                      New latching is always initiated
  Returns : <Result>  S_OK    if data latched (data in buffer if valid buffer)
                      S_FALSE if data not (yet) latched
            <Info>    Size of (required) buffer

  Descript: Return latched data and start new latching
  Notes   : Typically called first with a 'nil' pointer to start new data
            latching and to obtain the required size. Then called to actually
            obtain the data (note that the result indicates if data has been
            latched)
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetDebugBufferData(out Info: Word; Buffer: Pointer) : HRESULT;
begin
  // Always return required size
  Info := SizeOf(FLatchedData);
  // Always return status of latched data
  if FLatchData then
  begin
    // If not yet latched return
    Result := S_FALSE;
    Exit;
  end
  else
    Result := S_OK;
  // Buffer must 'exist' if we want to return data to it
  if Assigned(Buffer) then
    CopyMemory(Buffer, @FLatchedData, SizeOf(FLatchedData));
  // Initiate new latching
  FLatchData := True;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>    Number of buffers to store
                        Use '0' to retreive status only
  Returns : <Result>    S_OK    if data written
                        S_FALSE if data not (yet) written

  Descript: Write next buffers contents to files
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.SetDebugWriteBuffer(Buffers: Byte) : HRESULT;
begin
  // Always return status of written data
  if (FFileBuffers <> 0) then
  begin
    // If not yet latched return
    Result := S_FALSE;
    Exit;
  end
  else
    Result := S_OK;
  // Only initiate when non zero
  if Buffers <> 0 then
  begin
    FFilename :=  FormatDateTime('YYYYMMDD"T"HHMMSS', Now);
    FFileBuffersMax := Buffers;
    FFileBuffers    := Buffers;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <OnOff>    TRUE to activate log (new file will be created)
                       FALSE to stop log
  Returns : -

  Descript: (De)activate logging mechanism.
  Notes   : Activating an already started logging has no effect.
 ------------------------------------------------------------------------------}
procedure TMajorDvbPsi.SetDebugSetLog(OnOff: Boolean);
begin
  if not OnOff and Assigned(FLogStream) then
  begin
    ToLog('Log stopped');
    FreeAndNil(FLogStream);
  end;
  if OnOff and not Assigned(FLogStream) then
  begin
    FLogStream := TFileStream.Create(FormatDateTime('YYYYMMDD"T"HHMMSS', Now) + '.LOG', fmCreate);
    ToLog('Log started');
  end;
end;


{------------------------------------------------------------------------------
  Descript: <DvbFilter> interface. See there for the details.
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.DvbGetProgramName(ToGet: Word; out Info: PChar): HRESULT;
var
  DummyWord     : Word;
  DummyPids     : TDvbPids;
  DummyLanguages: TDvbLanguages;
  ResultString  : string;
begin
  Result := S_FALSE;
  if not Assigned(Info) then
    Exit;
  Result := S_OK;
  DvbFilter.DvbFilterGetProgramInfo(Lo(ToGet),
    DummyWord, DummyWord, DummyPids, DummyPids,
    DummyPids, DummyPids, DummyLanguages,
    DummyLanguages, DummyPids, DummyPids, ResultString);
  StrPCopy(Info, ResultString);
end;
function TMajorDvbPsi.DvbGetPid(StreamPacketData: PDvbTransportPacket; out Pid: Word): Boolean;
begin
  Result := DvbFilter.DvbFilterGetPid(StreamPacketData, Pid);
end;
function TMajorDvbPsi.DvbSetSignallingCallback(Pid: Word; hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallback(Pid, hWnd, Msg);
end;
function TMajorDvbPsi.DvbSetSignallingCallbackPmt(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPmt(hWnd, Msg);
end;
function TMajorDvbPsi.DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackEvent(hWnd, Msg);
end;
function TMajorDvbPsi.DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPidFilter(Pid, Filter);
end;
function TMajorDvbPsi.DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := DvbFilter.DvbFilterGetPidFilter(Pid);
end;
function TMajorDvbPsi.DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPsiFilter(Psi, Filter);
end;
function TMajorDvbPsi.DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := DvbFilter.DvbFilterGetPsiFilter(Psi);
end;
function TMajorDvbPsi.DvbGetProgramInfo(ProgramNumber: Byte;
                            out PmtPid: Word;
                            out PcrPid: Word;
                            out VideoPids: TDvbPids;
                            out AudioPids: TDvbPids;
                            out TeletextPids: TDvbPids;
                            out SubtitlePids: TDvbPids;
                            out AudioLanguages: TDvbLanguages;
                            out SubtitleLanguages: TDvbLanguages;
                            out EcmPids: TDvbPids;
                            out CaIds: TDvbPids;
                            out ProgramName: PChar): Boolean;
var
  ProgName: string;
begin
  Result := DvbFilter.DvbFilterGetProgramInfo(ProgramNumber,
              PmtPid, PcrPid, VideoPids, AudioPids,
              TeletextPids, SubtitlePids, AudioLanguages,
              SubtitleLanguages, EcmPids, CaIds, ProgName);
  StrPCopy(ProgramName, ProgName);              
end;
function  TMajorDvbPsi.DvbGetEventInfo(ProgramNumber: Byte; Present: Boolean; Events: TDvbEventSections ): Boolean;
begin
  Result := DvbFilter.DvbFilterGetEventInfo(ProgramNumber, Present, Events);
end;
function  TMajorDvbPsi.DvbGetNumberOfPrograms: Byte;
begin
  Result := DvbFilter.DvbFilterGetNumberOfPrograms;
end;
function  TMajorDvbPsi.DvbGetErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetErrors;
end;
function  TMajorDvbPsi.DvbGetPacketSyncErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetPacketSyncErrors;
end;
procedure TMajorDvbPsi.DvbResetErrors;
begin
  DvbFilter.DvbFilterResetErrors;
end;
procedure TMajorDvbPsi.DvbCreateTables;
begin
  DvbFilter.DvbFilterCreateTables;
end;
{------------------------------------------------------------------------------
  Descript: End of <DvbFilter> interface
 ------------------------------------------------------------------------------}


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  S_FALSE if does not need saving
                      S_OK    if needs saving

  Descript: Indicate if a property has changed.
  Notes   : <IPersistStream>
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.IsDirty: HRESULT;
begin
  Result := S_FALSE;
end;


{------------------------------------------------------------------------------
  Params  : <stm>     Stream data
  Returns : <Result>

  Descript: Load data from stream.
  Notes   : <IPersistStream>
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.Load(const stm: IStream): HRESULT;
begin
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : <stm>          Stream data
            <fClearDirty>  True reset dirty flag
  Returns : <Result>

  Descript: Save data to stream.
  Notes   : <IPersistStream>
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.Save(const stm: IStream; fClearDirty: BOOL): HRESULT;
begin
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  :
  Returns : <Result>
            <cbSize>  Stream size

  Descript: Needed bytes for stream.
  Notes   : <IPersistStream>
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.GetSizeMax(out cbSize: Largeint): HRESULT;
begin
  Result := S_OK;
  cbSize := 0;
end;


{------------------------------------------------------------------------------
  Params  : <IID>
  Returns : <Result>
            <Obj>

  Descript: Support for delegating and non delegating IUnknown interfaces.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.NonDelegatingQueryInterface(const IID: TGUID; out Obj): HRESULT;
begin
  Result := E_NOINTERFACE;
  // Since we are dealing with 16 byte long types we can not just use '=' on it
  // without typecasting or using <IsEqualGUID> (or simular)
  if IsEqualGUID(IID, T_IID_MajorDvbPsi) then
  begin
    if GetInterface(IMajorDvbPsi, Obj) then
    begin
//      ToLog('<NonDelegatingQueryInterface> IMajorDvbPsi success');
      Result := S_OK;
    end
    else
      ToLog('<NonDelegatingQueryInterface> IMajorDvbPsi failed');
    Exit;
  end;
  if IsEqualGUID(IID, IID_ISpecifyPropertyPages) then
  begin
    if GetInterface(ISpecifyPropertyPages, Obj) then
    begin
//      ToLog('<NonDelegatingQueryInterface> ISpecifyPropertyPages success');
      Result := S_OK;
    end
    else
      ToLog('<NonDelegatingQueryInterface> ISpecifyPropertyPages failed');
    Exit;
  end;
  if IsEqualGUID(IID, IID_IPersistStream) then
  begin
    if GetInterface(IPersistStream, Obj) then
    begin
//      ToLog('<NonDelegatingQueryInterface> IPersistStream success');
      Result := S_OK;
    end
    else
      ToLog('<NonDelegatingQueryInterface> IPersistStream failed');
    Exit;
  end;
  Result := inherited NonDelegatingQueryInterface(IID, Obj);
end;


{------------------------------------------------------------------------------
  Params  : <Sample>  Input data
  Returns : -

  Descript: Handle data from input pin.
  Notes   :
 ------------------------------------------------------------------------------}
function TMajorDvbPsi.Transform(Sample: IMediaSample): HRESULT;
type
  PTransportPackets = ^TTransportPackets;
  TTransportPackets = packed array[0..$FFFF] of TDvbTransportPacket;
var
  StreamData: PTransportPackets;
  Packets   : Word;
  Loop      : Word;
  Pid       : Word;
  Filter    : TDvbPacketFilter;
  SampleSize: LongWord;
  Filename  : string;
  FileStream: TFileStream;
begin
  if FDebugCountDsFilterIn = $FFFF then
    FDebugCountDsFilterIn := 0
  else
    Inc(FDebugCountDsFilterIn);

  // Get pointer to data
  Sample.GetPointer(PByte(StreamData));
  if Assigned(StreamData) then
  begin
    // Get size of sample
    // We use the smallest of the reported buffer and data size
    FValidDataSize := Sample.GetActualDataLength;   //Get Size of Data
    FBufferSize    := Sample.GetSize;
    SampleSize     := FValidDataSize;

    // Latch some data if requested, but only if correct size
    if FLatchData and (SampleSize > SizeOf(FLatchedData)) then
    begin
      CopyMemory(@FLatchedData, StreamData, SizeOf(FLatchedData));
      FLatchData := False;
    end;

    // Write data to file if requested, but only if size > 0
    if (FFileBuffers <> 0) and (SampleSize > 0) then
    begin
      // Create name for base file and index
      Filename := FFilename + format('_%d.DVB', [FFileBuffersMax-FFileBuffers]);
      try
        FileStream := TFileStream.Create(Filename, fmCreate);
        try
          FileStream.Write(StreamData^, SampleSize);
        finally
          FileStream.Free;
        end;
      except
      end;
      Dec(FFileBuffers);
    end;

    // We should have a multitude of single packets
    Packets := SampleSize div CDvbPacketSize;
    if (SampleSize mod CDvbPacketSize) = 0 then
    begin
      if FDebugCountDsFilterInSizeCorrect = $FFFF then
        FDebugCountDsFilterInSizeCorrect := 0
      else
        Inc(FDebugCountDsFilterInSizeCorrect);
    end;
    // Process 'whole' packets
    if Packets > 0 then
    begin
      // Now go through all packets and process them accoring to the
      // type of packet
      for Loop := 0 to Packets-1 do
      begin
        if FDebugCountPackets = $FFFF then
          FDebugCountPackets := 0
        else
          Inc(FDebugCountPackets);

        // The Pid is the identifier of what type of information we are dealing with
        if DvbGetPid(@StreamData[Loop], Pid) then
        begin
          if FDebugCountValidSyncs = $FFFF then
            FDebugCountValidSyncs := 0
          else
            Inc(FDebugCountValidSyncs);

          Filter := DvbFilter.DvbFilterGetPidFilter(Pid);
          if Assigned(@Filter) then
          begin
            if FDebugCountToPidFilter = $FFFF then
              FDebugCountToPidFilter := 0
            else
              Inc(FDebugCountToPidFilter);
            Filter(@StreamData[Loop]);
          end;
        end
        else
        begin
          if FDebugCountFalseSyncs = $FFFF then
            FDebugCountFalseSyncs := 0
          else
            Inc(FDebugCountFalseSyncs);
        end;
      end;
    end;
  end;

  // Deliver this sample (S_FALSE if not to deliver)
  Result := S_OK;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}
initialization
  InstanceCount := 0;

  TBCClassFactory.CreateFilter(TMajorDvbPsi, 'MajorDvb PSI Parser', CLSID_MajorDvbPsi,
    CLSID_LegacyAmFilterCategory, MERIT_DO_NOT_USE, 2, @SudPins);

end.
