{******************************************************************************
  FileName............: Example.pas
  Copyright...........: 2025-2025 by M.Majoor
 ------------------------------------------------------------------------------
  Example for using the Sagebox DLL

  The MIT License (MIT)

  Copyright (c) 2025 Marcel Majoor

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.


  Date/Version   Comment
   2025.12.07    - Initial release
 ******************************************************************************}
program Example;

uses
  uSageboxTypes   in '..\uSageboxTypes.pas',
  uSageboxLibrary in '..\uSageboxLibrary.pas',
  uSageboxHelpers in '..\uSageboxHelpers.pas',
  System.Sysutils;

const
  CColors: array[0..2] of TSbString = ( 'green', 'yellow', 'purple' );

type
  TTriangleColors = array[0..2] of TSbColors;


// The ASliderVaraint will use a slightly different color achitecture (fixed color triangle, not related to depth)
procedure CalcTriangle(AWindow: TSbLong; ASliderVariant: TSbBool; AColors: TTriangleColors; var AColor: TSbColors; ALevel: TSbInt; ATopPoint: TSbCPoint; ALeftPoint: TSbCPoint; ARightPoint: TSbCPoint; ADepth: TSbInt);
var
  LLeftMid  : TSbCPoint;
  LRightMid : TSbCPoint;
  LBottomMid: TSbCPoint;
begin
  // See if we should stop
  if (ALevel = 0) then
    _SageVB_Window_DrawTriangle_Rgb(AWindow, ATopPoint.X, ATopPoint.Y, ARightPoint.X, ARightPoint.Y, ALeftPoint.X, ALeftPoint.Y, AColor, True, False, 0)
  else
  begin
    // Find the edge midpoints.
    LLeftMid   := TSbCPoint.Create((ATopPoint.X  + ALeftPoint.X ) / 2.0, (ATopPoint.Y  + ALeftPoint.Y ) / 2.0);
    LRightMid  := TSbCPoint.Create((ATopPoint.X  + ARightPoint.X) / 2.0, (ATopPoint.Y  + ARightPoint.Y) / 2.0);
    LBottomMid := TSbCPoint.Create((ALeftPoint.X + ARightPoint.X) / 2.0, (ALeftPoint.Y + ARightPoint.Y) / 2.0);
    // Recursively draw smaller triangles.
    if ASliderVariant then
      AColor := AColors[0]
    else
      if (ADepth = 0) then AColor := AColors[0];
    CalcTriangle(AWindow, ASliderVariant, AColors, AColor, ALevel - 1, ATopPoint, LLeftMid  , LRightMid  , ADepth+1);
    if ASliderVariant then
      AColor := AColors[1]
    else
      if (ADepth = 0) then AColor := AColors[1];
    CalcTriangle(AWindow, ASliderVariant, AColors, AColor, ALevel - 1, LLeftMid , ALeftPoint, LBottomMid , ADepth+1);
    if ASliderVariant then
      AColor := AColors[2]
    else
      if (ADepth = 0) then AColor := AColors[2];
    CalcTriangle(AWindow, ASliderVariant, AColors, AColor, ALevel - 1, LRightMid , LBottomMid, ARightPoint, ADepth+1);
  end;
end;


procedure Main();
var
  LWindow        : TSbLong;
  LSlider        : TSbLong;
  LInputBox      : TSbLong;
  LLevel         : TSbInt;
  LP1            : TSbCPoint;
  LP2            : TSbCPoint;
  LP3            : TSbCPoint;
  LColors        : TSbColors;
  LCenter        : TSbCPoint;
  LReturnPressed : TSbBool;
  LOldSliderPos  : TSbLong;
  LTriangleColors: TTriangleColors;
  LTriangleColor : TSbColors;
begin
  LP1     := TSbCPoint.Create(   0, -275);
  LP2     := TSbCPoint.Create( 500,  375);
  LP3     := TSbCPoint.Create(-500,  375);
  LCenter := TSbCPoint.Create(   0,    0);
  LLevel  := 5;
  LOldSliderPos := LLevel;
  //
  LWindow := _SageVB_NewWindow(0, 'Sagebox Windows Hello World -- ');
  // Set colors
  _SageVB_ColorFromString(CColors[0], LTriangleColors[0].color1, LTriangleColors[0].color2);
  _SageVB_ColorFromString(CColors[1], LTriangleColors[1].color1, LTriangleColors[1].color2);
  _SageVB_ColorFromString(CColors[2], LTriangleColors[2].color1, LTriangleColors[2].color2);
  LTriangleColor := LTriangleColors[0];

  LSlider   := _SageVB_DevSlider  ('Select Level (0-10)',                                         TSbKw.OperatorPlus(TSbKw.Range(0, 10), TSbKw.Default(LLevel)).Pointer, False);
  LInputBox := _SageVB_DevInputBox('Enter Level (0-10)' , TSbKw.OperatorPlus(TSbKw.NumbersOnly(), TSbKw.OperatorPlus(TSbKw.Range(0, 10), TSbKw.Default(LLevel))).Pointer);

  _SageVB_ColorFromString('darkblue,black', LColors.color1, LColors.color2);
  _SageVB_Window_Cls(LWIndow, LColors.color1.red, LColors.color1.green, LColors.color1.blue, LColors.color2.red, LColors.color2.green, LColors.color2.blue, True);

  _SageVB_Window_GetWindowCenter(LWindow, LCenter.X, LCenter.Y);

  _SageVB_Write('{60}Enter or drag level' + #10 + 'to draw Sierpinski Triangle',  LWindow, TSbKw.JustCenter().Pointer);

  while _SageVB_GetEvent() do
  begin
    // ReturnPressed is a single 'event' so keep record of that
    LReturnPressed := _SageVB_InputBox_ReturnPressed(LInputBox);
    if LReturnPressed or (_SageVB_Slider_GetPos(LSlider) <> LOldSliderPos) then
    begin
      // There seem to be no 'setters' so we can not keep the two widgets in sync ...
      if LReturnPressed then
        LLevel := _SageVB_InputBox_GetInteger(LInputBox)
      else
      begin
        LLevel        := _SageVB_Slider_GetPos(LSlider);
        LOldSliderPos := LLevel;
      end;
      _SageVB_Window_Cls(LWindow, -1, -1, -1, -1, -1, -1, False);
      _SageVB_Write(AnsiString(Format('{40}Delphi Sagebox DLL' + #10 + 'Sierpinski Triangle' + #10 + '{15}{cyan}Level %s', [IntToStr(LLevel)])), LWindow, TSbKw.OperatorPlus(TSbKw.JustTopCenter(), TSbKw.PadY(20)).Pointer);
      CalcTriangle(LWindow, not LReturnPressed, LTriangleColors, LTriangleColor, LLevel, TSbCPoint.OperatorPlus(LP1, LCenter), TSbCPoint.OperatorPlus(LP2, LCenter), TSbCPoint.OperatorPlus(LP3, LCenter), 0);
      if LReturnPressed then
        _SageVB_InputBox_ClearText(LInputBox);
    end;
  end;
  _SageVB_ExitButton('ExitButton');
end;


begin
  Main();
end.

